package org.qas.qtest.api.services.execution.model;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.qas.qtest.api.internal.model.FieldValue;
import org.qas.qtest.api.internal.model.QTestBaseModel;
import org.qas.qtest.api.services.design.model.TestCase;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * TestRun
 *
 * @author Dzung Nguyen
 * @version $Id TestRun 2014-03-28 07:08:30z dungvnguyen $
 * @since 1.0
 */
public class TestRun extends QTestBaseModel<TestRun> {
  @JsonProperty("id")
  private Long id;

  @JsonProperty("name")
  private String name;

  @JsonProperty("properties")
  private List<FieldValue> fieldValues;

  @JsonProperty("test_case_version_id")
  private Long testCaseVersionId;

  @JsonProperty("order")
  private Integer order;

  @JsonProperty("pid")
  private String pid;

  @JsonProperty("test_case")
  private TestCase testCase;

  public TestRun() {
  }

  /**
   * @return the test run identifier.
   */
  public Long getId() {
    return id;
  }

  /**
   * Sets test run id.
   *
   * @param id the given test run identifier to set.
   */
  public TestRun setId(Long id) {
    this.id = id;
    return this;
  }

  /**
   * Sets test run id.
   *
   * @param id the given test run identifier to set.
   */
  public TestRun withId(Long id) {
    setId(id);
    return this;
  }

  /**
   * @return the test run name.
   */
  public String getName() {
    return name;
  }

  /**
   * Sets the test run name.
   *
   * @param name the given name of test run.
   */
  public TestRun setName(String name) {
    this.name = name;
    return this;
  }

  /**
   * Sets the test run name.
   *
   * @param name the given name of test run.
   */
  public TestRun withName(String name) {
    setName(name);
    return this;
  }

  /**
   * @return the list of field value objects.
   */
  public List<FieldValue> getFieldValues() {
    if (null == fieldValues) {
      return Collections.emptyList();
    }
    return fieldValues;
  }

  /**
   * Sets the list of field value objects.
   *
   * @param values the given field value objects to set.
   */
  public TestRun setFieldValues(List<FieldValue> values) {
    this.fieldValues = new ArrayList<>();
    if (null == values) {
      return this;
    }
    for (FieldValue fieldValue : values) {
      addFieldValue(fieldValue);
    }
    return this;
  }

  /**
   * Sets the list of field value objects.
   *
   * @param values the given field value objects to set.
   * @return the test-run instance.
   */
  public TestRun withFieldValues(List<FieldValue> values) {
    setFieldValues(values);
    return this;
  }

  /**
   * Adds the property value to requirement and return itself.
   *
   * @param value the given field value to add.
   * @return the test-run instance.
   */
  public TestRun addFieldValue(FieldValue value) {
    if (value == null || value.getValue() == null) {
      return this;
    }
    if (fieldValues == null) {
      fieldValues = new ArrayList<>();
    }
    fieldValues.add(value);
    return this;
  }

  /**
   * @return the test case version id.
   */
  public Long getTestCaseVersionId() {
    return testCaseVersionId;
  }

  /**
   * Sets the test case version id.
   *
   * @param testCaseVersionId the given test case version id to set.
   */
  public TestRun setTestCaseVersionId(Long testCaseVersionId) {
    this.testCaseVersionId = testCaseVersionId;
    return this;
  }

  /**
   * Sets the test case version id.
   *
   * @param testCaseVersionId the given test case version id to set.
   */
  public TestRun withTestCaseVersionId(Long testCaseVersionId) {
    setTestCaseVersionId(testCaseVersionId);
    return this;
  }

  /**
   * @return the test run order.
   */
  public Integer getOrder() {
    return order;
  }

  /**
   * Sets the test run order.
   *
   * @param order the given test run order to set.
   */
  public TestRun setOrder(Integer order) {
    this.order = order;
    return this;
  }

  /**
   * Sets the test run order.
   *
   * @param order the given test run order to set.
   */
  public TestRun withOrder(Integer order) {
    setOrder(order);
    return this;
  }

  /**
   * @return the test-run pid.
   */
  public String getPid() {
    return pid;
  }

  /**
   * Sets test-run pid.
   *
   * @param pid the given test-run pid.
   */
  public TestRun setPid(String pid) {
    this.pid = pid;
    return this;
  }

  /**
   * Sets test-run pid.
   *
   * @param pid the given test-run pid.
   * @return current {@link TestRun test-run} instance.
   */
  public TestRun withPid(String pid) {
    setPid(pid);
    return this;
  }

  /**
   * @return the associated test case.
   */
  public TestCase getTestCase() {
    return testCase;
  }

  /**
   * Sets test case.
   *
   * @param testCase the given test case instance to set.
   */
  public TestRun setTestCase(TestCase testCase) {
    this.testCase = testCase;
    return this;
  }

  /**
   * Sets test case.
   *
   * @param testCase the given test case instance to set.
   */
  public TestRun withTestCase(TestCase testCase) {
    setTestCase(testCase);
    return this;
  }

  @Override
  protected TestRun clone() {
    TestRun that = new TestRun();

    that.setPropertiesFrom(this);

    return that;
  }

  @Override
  public String elementName() {
    return "test-run";
  }

  @Override
  public String jsonElementName() {
    return "test_run";
  }
}
