package org.qas.qtest.api.services.defect;

import org.qas.api.AuthServiceException;
import org.qas.api.ClientConfiguration;
import org.qas.api.handler.AsyncHandler;
import org.qas.qtest.api.auth.DefaultQTestCredentialsProviderChain;
import org.qas.qtest.api.auth.QTestCredentials;
import org.qas.qtest.api.auth.QTestCredentialsProvider;
import org.qas.qtest.api.auth.StaticQTestCredentialsProvider;
import org.qas.qtest.api.internal.model.CreateObjectCommentRequest;
import org.qas.qtest.api.services.defect.model.CreateDefectRequest;
import org.qas.qtest.api.services.defect.model.Defect;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * DefectServiceAsyncClient
 *
 * @author Dzung Nguyen
 * @version $Id DefectServiceAsyncClient 2014-07-16 17:39:30z dungvnguyen $
 * @since 1.0
 */
public class DefectServiceAsyncClient extends DefectServiceClient
  implements DefectServiceAsync {
  /**
   * Constructs a new client to invoke service method on DefectService using
   * the default qTest credentials provider and default client configuration options.
   */
  public DefectServiceAsyncClient() {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on DefectService using
   * the default qTest credentials provider and default client configuration options.
   *
   * @param executorService the executor service for executing asynchronous request.
   */
  public DefectServiceAsyncClient(ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on DefectService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to DefectService
   */
  public DefectServiceAsyncClient(ClientConfiguration clientConfiguration) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on DefectService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to DefectService.
   * @param executorService the executor service for executing asynchronous request.
   */
  public DefectServiceAsyncClient(ClientConfiguration clientConfiguration, ExecutorService executorService) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on DefectService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   */
  public DefectServiceAsyncClient(QTestCredentials credentials) {
    this(credentials, new ClientConfiguration(), Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on DefectService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param executorService the executor service for executing asynchronous request.
   */
  public DefectServiceAsyncClient(QTestCredentials credentials, ExecutorService executorService) {
    this(credentials, new ClientConfiguration(), executorService);
  }

  /**
   * Constructs a new client to invoke service method on DefectService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to DefectService
   */
  public DefectServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration) {
    this(credentials, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on DefectService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to DefectService
   * @param executorService the executor service for executing asynchronous request.
   */
  public DefectServiceAsyncClient(QTestCredentials credentials, ClientConfiguration clientConfiguration,
                                       ExecutorService executorService) {
    this(new StaticQTestCredentialsProvider(credentials), clientConfiguration, executorService);
  }

  /**
   * Constructs a new client to invoke service method on DefectService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to DefectService
   */
  public DefectServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                       ClientConfiguration clientConfiguration) {
    this(credentialsProvider, clientConfiguration, Executors.newCachedThreadPool());
  }

  /**
   * Constructs a new client to invoke service method on DefectService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to DefectService
   * @param executorService the executor service for executing asynchronous request.
   */
  public DefectServiceAsyncClient(QTestCredentialsProvider credentialsProvider,
                                       ClientConfiguration clientConfiguration, ExecutorService executorService) {
    super(credentialsProvider, clientConfiguration);
    this.executorService = executorService;
  }

  @Override
  public Future<Defect> createDefectAsync(final CreateDefectRequest createDefectRequest) throws AuthServiceException {
    return executorService.submit(new Callable<Defect>() {
      @Override
      public Defect call() throws Exception {
        return createDefect(createDefectRequest);
      }
    });
  }

  @Override
  public Future<Defect> createDefectAsync(final CreateDefectRequest createDefectRequest,
                                          final AsyncHandler<CreateDefectRequest, Defect> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<Defect>() {
      @Override
      public Defect call() throws Exception {
        final Defect result;

        try {
          result = createDefect(createDefectRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(createDefectRequest, result);
        return result;
      }
    });
  }

  @Override
  public Future<Defect> addCommentAsync(final CreateObjectCommentRequest createObjectCommentRequest) throws AuthServiceException {
    return executorService.submit(new Callable<Defect>() {
      @Override
      public Defect call() throws Exception {
        return addComment(createObjectCommentRequest);
      }
    });
  }

  @Override
  public Future<Defect> addCommentAsync(final CreateObjectCommentRequest createObjectCommentRequest,
                                        final AsyncHandler<CreateObjectCommentRequest, Defect> asyncHandler) throws AuthServiceException {
    return executorService.submit(new Callable<Defect>() {
      @Override
      public Defect call() throws Exception {
        Defect result;

        try {
          result = addComment(createObjectCommentRequest);
        } catch (Exception ex) {
          asyncHandler.onError(ex);
          throw ex;
        }

        asyncHandler.onSuccess(createObjectCommentRequest, result);
        return result;
      }
    });
  }

  @Override
  public void shutdown() {
    super.shutdown();
    executorService.shutdown();
  }
}
