package org.qas.qtest.api.services.authenticate.model;

import org.qas.api.ApiServiceRequest;
import org.qas.api.StringUtil;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * OAuthAuthenticateRequest
 *
 * @author Dzung Nguyen
 * @version $Id OAuthAuthenticateRequest 2016-02-26 14:31:30z dzungvnguyen $
 * @since 1.0
 */
public abstract class OAuthAuthenticateRequest extends ApiServiceRequest {
  private static final String BASIC_AUTH_TOKEN = StringUtil.base64Encode("qtest-java-sdk:");
  //~ class properties ========================================================
  protected final OAuthGrantType grantType;
  protected final String token;

  //~ class members ===========================================================

  /**
   * Creates {@link OAuthAuthenticateRequest} instance from the given
   * {@link OAuthGrantType OAuth grant type} value.
   *
   * @param grantType the given {@link OAuthGrantType OAuth grant type} value.
   * @param token     the given authentication token request.
   */
  protected OAuthAuthenticateRequest(OAuthGrantType grantType, String token) {
    this.grantType = grantType;
    this.token = token;
  }

  /**
   * @return the {@link OAuthGrantType OAuth grant type} value.
   */
  public OAuthGrantType grantType() {
    return this.grantType;
  }

  /**
   * @return the token information.
   */
  public String getToken() {
    return token;
  }

  /**
   * Obtains the request parameters.
   *
   * @return the request parameters.
   */
  public Map<String, String> requestParams() {
    HashMap<String, String> requestParams = new HashMap<String, String>();

    requestParams.put("grant_type", grantType.type());

    appendInternalRequestParams(requestParams);
    return Collections.unmodifiableMap(requestParams);
  }

  /**
   * Appends the internal request parameters.
   *
   * @param requestParams the map of request parameters.
   */
  protected abstract void appendInternalRequestParams(HashMap<String, String> requestParams);
  //~ class helpers ===========================================================

  /**
   * Define the supported OAuth grant type.
   */
  public enum OAuthGrantType {
    PASSWORD("password"),
    CLIENT_CREDENTIALS("client_credentials"),
    REFRESH_TOKEN("refresh_token");

    //~ class properties ======================================================
    private String grantType;

    //~ class members =========================================================
    OAuthGrantType(String grantType) {
      this.grantType = grantType;
    }

    /**
     * @return the OAuth grant type.
     */
    public String type() {
      return this.grantType;
    }
  }

  /**
   * PasswordAuthenticateRequest
   *
   * @author Dzung Nguyen
   * @version $Id PasswordAuthenticateRequest 2016-03-07 14:31:30z dzungvnguyen $
   * @since 1.0
   */
  public static final class PasswordAuthenticateRequest extends OAuthAuthenticateRequest {
    //~ class properties ======================================================
    private final String username;
    private final String password;

    //~ class members =========================================================

    /**
     * Creates {@link PasswordAuthenticateRequest} from the given basic token,
     * username and password.
     *
     * @param username the given username.
     * @param password the given password.
     */
    public PasswordAuthenticateRequest(String username, String password) {
      super(OAuthGrantType.PASSWORD, BASIC_AUTH_TOKEN);
      this.username = username;
      this.password = password;
    }

    /**
     * Creates {@link PasswordAuthenticateRequest} from the given basic token,
     * username and password.
     *
     * @param token    the given basic token.
     * @param username the given username.
     * @param password the given password.
     */
    public PasswordAuthenticateRequest(String token, String username, String password) {
      super(OAuthGrantType.PASSWORD, token);
      this.username = username;
      this.password = password;
    }

    /**
     * @return the current username.
     */
    public String getUsername() {
      return username;
    }

    /**
     * @return the current password.
     */
    public String getPassword() {
      return password;
    }

    @Override
    protected void appendInternalRequestParams(HashMap<String, String> requestParams) {
      requestParams.put("username", getUsername());
      requestParams.put("password", getPassword());
    }

    @Override
    public String toString() {
      final StringBuilder sb = new StringBuilder("PasswordAuthenticateRequest{");
      sb.append("grantType=").append(grantType.type());
      sb.append(", username='").append(username).append('\'');
      sb.append(", password='").append(password).append('\'');
      sb.append('}');
      return sb.toString();
    }
  }

  /**
   * ClientCredentialsAuthenticateRequest
   *
   * @author Dzung Nguyen
   * @version $Id ClientCredentialsAuthenticateRequest 2016-03-07 17:36:30z dzungvnguyen $
   * @since 1.0
   */
  public static final class ClientCredentialsAuthenticateRequest extends OAuthAuthenticateRequest {
    //~ class members =========================================================

    /**
     * ClientCredentialsAuthenticateRequest
     *
     * @param token the given basic token to set.
     */
    public ClientCredentialsAuthenticateRequest(String token) {
      super(OAuthGrantType.CLIENT_CREDENTIALS, token);
    }

    @Override
    protected void appendInternalRequestParams(HashMap<String, String> requestParams) {
    }

    @Override
    public String toString() {
      final StringBuilder sb = new StringBuilder("ClientCredentialsAuthenticateRequest{");
      sb.append("grantType=").append(grantType.type());
      sb.append(", token='").append(token).append('\'');
      sb.append('}');
      return sb.toString();
    }
  }

  /**
   * RefreshTokenAuthenticateRequest
   *
   * @author Dzung Nguyen
   * @version $Id RefreshTokenAuthenticateRequest 2016-03-07 17:36:30z dzungvnguyen $
   * @since 1.0
   */
  public static final class RefreshTokenAuthenticateRequest extends OAuthAuthenticateRequest {
    //~ class properties ======================================================
    private final String refreshToken;

    //~ class members =========================================================
    /**
     * Creates {@link RefreshTokenAuthenticateRequest} instance from the given basic token
     * and refresh token.
     *
     * @param token        the given basic token.
     * @param refreshToken the given refresh token.
     */
    public RefreshTokenAuthenticateRequest(String token, String refreshToken) {
      super(OAuthGrantType.REFRESH_TOKEN, token);
      this.refreshToken = refreshToken;
    }

    /**
     * @return the refresh token.
     */
    public String getRefreshToken() {
      return refreshToken;
    }

    @Override
    protected void appendInternalRequestParams(HashMap<String, String> requestParams) {
      requestParams.put("refresh_token", getRefreshToken());
    }

    @Override
    public String toString() {
      final StringBuilder sb = new StringBuilder("RefreshTokenAuthenticateRequest{");
      sb.append("grantType=").append(grantType.type());
      sb.append(", token='").append(token).append('\'');
      sb.append(", refreshToken='").append(refreshToken).append('\'');
      sb.append('}');
      return sb.toString();
    }
  }
}
