package org.qas.qtest.api.auth;

import org.qas.api.AuthClientException;
import org.qas.api.Request;
import org.qas.api.internal.util.Encoders;
import org.qas.api.internal.util.Https;
import org.qas.api.internal.util.google.net.HttpHeaders;

import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.SimpleTimeZone;

/**
 * QTest1Signer
 *
 * @author Dzung Nguyen
 * @version $Id QTest1Signer 2014-03-27 14:44:30z dungvnguyen $
 * @since 1.0
 */
public class QTest1Signer extends AbstractQTestSigner {
  //~ class members ===========================================================
  @Override
  protected void sign(Request request, QTestCredentials credentials) {
    // add host to header.
    addHostHeader(request);

    // compute the signature.
    Date date = getDateFromRequest(request);

    // add date to header.
    request.setHeader("x-qtest-date", formatDateTime(date));

    String contentSha256 = request.getHeaders().get("x-qtest-content-sha256");
    if (contentSha256 == null || "required".equals(contentSha256)) {
      contentSha256 = calculateContentHash(request);
    }
    request.setHeader("x-qtest-content-sha256", contentSha256);

    // set authorization header.
    request.setHeader(HttpHeaders.AUTHORIZATION, credentials.getToken());
  }

  protected void addHostHeader(Request request) {
    String hostHeader = request.getEndpoint().getHost();

    if (Https.isUsingNonDefaultPort(request.getEndpoint())) {
      hostHeader += ":" + request.getEndpoint().getPort();
    }

    request.setHeader(HttpHeaders.HOST, hostHeader);
  }

  protected String formatDateTime(Date date) {
    SimpleDateFormat dateTimeFormat = new SimpleDateFormat("yyyyMMdd'T'HHmmssZ");
    dateTimeFormat.setTimeZone(new SimpleTimeZone(0, "UTC"));
    return dateTimeFormat.format(date);
  }

  private Date getDateFromRequest(Request request) {
    return getSignatureDate(request.getTimeOffset());
  }

  protected String calculateContentHash(Request request) throws AuthClientException {
    InputStream payloadStream = getBinaryRequestPayloadStream(request);
    payloadStream.mark(Integer.MAX_VALUE);
    String contentSha256 = Encoders.bytesToHex(hash(payloadStream));

    try {
      payloadStream.reset();
    } catch (IOException ex) {
      throw new AuthClientException("Unable to reset stream after calculating qTest1 signature", ex);
    }

    return contentSha256;
  }
}
