package org.qas.api.transform;

import org.qas.api.ApiServiceRequest;
import org.qas.api.Constants;
import org.qas.api.Request;
import org.qas.api.http.HttpMethod;
import org.qas.api.internal.util.StringInputStream;
import org.qas.api.internal.util.UniqueKeys;
import org.qas.api.internal.util.google.net.HttpHeaders;
import org.qas.api.support.RequestSupport;
import org.qas.qtest.api.services.common.PagingRequest;

/**
 * @author trongle
 * @version $Id 4/21/2017 4:33 PM
 */
public abstract class AbstractMarshaller<T extends Request, R extends ApiServiceRequest> implements Marshaller<T, R> {
  
  @Override
  public T marshall(R in) throws Exception {
    validateRequest(in);
    
    // create request.
    Request request = new RequestSupport(in, getServiceName());
    
    // set additional information.
    request.setHeader(Constants.Header.X_ACTION, getActionName());
    request.setHeader(Constants.Header.X_API_VERSION, "V3");
    request.setHeader(Constants.Header.X_NONCE, UniqueKeys.generateNonce());
    request.setHeader(Constants.Header.X_SERVICE_NAME, getServiceName());
    
    // set content type.
    request.setHeader(HttpHeaders.CONTENT_TYPE, Constants.ContentType.JSON);
    request.setHeader(HttpHeaders.ACCEPT, Constants.ContentType.JSON);
    
    StringBuilder rpBuilder = getResourcePath(in, new StringBuilder());
    request.setResourcePath(rpBuilder.toString());
    // set content.
    String content = getContent(in);
    if (null != content) {
      request.setContent(new StringInputStream(content));
    }
    build((T) request, in);
    return (T) request;
  }
  
  public boolean isParseJson() {
    return true;
  }
  
  protected abstract void validateRequest(R in) throws Exception;
  
  /**
   * More action
   *
   * @param request request
   * @param in in
   */
  protected void build(T request, R in) {
    request.setHttpMethod(HttpMethod.GET);
  }
  
  protected abstract String getServiceName();
  
  protected abstract String getActionName();
  
  protected abstract String getContent(R in) throws Exception;
  
  protected abstract StringBuilder getResourcePath(R in, StringBuilder rpBuilder);
  
  protected StringBuilder buildPaging(StringBuilder rpBuilder, PagingRequest in) {
    if (in.getPage() > 0) {
      rpBuilder.append("?page=" + in.getPage());
      if (in.getPageSize() > 0) {
        rpBuilder.append("&pageSize=" + in.getPageSize());
      }
    } else {
      if (in.getPageSize() > 0) {
        rpBuilder.append("?pageSize=" + in.getPageSize());
      }
    }
    return rpBuilder;
  }
}
