package org.qas.api.transform;

import org.qas.api.AuthServiceException;

import java.lang.reflect.Constructor;

/**
 * AbstractErrorUnmarshaller
 *
 * @author Dzung Nguyen
 * @version $Id AbstractErrorUnmarshaller 2014-03-27 11:12:30z dungvnguyen $
 * @since 1.0
 */
public abstract class AbstractErrorUnmarshaller<T> implements Unmarshaller<AuthServiceException, T> {
  //~ class properties ========================================================
  /**
   * The type of AuthServiceException that will be instantiated. Subclasses
   * specialized for a specific type of exception can control this through the
   * protected constructor.
   */
  protected final Class<? extends AuthServiceException> exceptionClass;

  //~ class members ===========================================================
  /**
   * Constructs a new error unmarshaller that will unmarshall error responses
   * into AuthServiceException objects.
   */
  public AbstractErrorUnmarshaller() {
    this(AuthServiceException.class);
  }

  /**
   * Constructs a new error unmarshaller that will unmarshall error responses
   * into objects of the specified class, extending AuthServiceException.
   *
   * @param exceptionClass
   *            The subclass of AuthServiceException which will be
   *            instantiated and populated by this class.
   */
  public AbstractErrorUnmarshaller(Class<? extends AuthServiceException> exceptionClass) {
    this.exceptionClass = exceptionClass;
  }

  /**
   * Constructs a new exception object of the type specified in this class's
   * constructor and sets the specified error message.
   *
   * @param message
   *            The error message to set in the new exception object.
   *
   * @return A new exception object of the type specified in this class's
   *         constructor and sets the specified error message.
   *
   * @throws Exception
   *             If there are any problems using reflection to invoke the
   *             exception class's constructor.
   */
  protected AuthServiceException newException(String message) throws Exception {
    Constructor<? extends AuthServiceException> constructor = exceptionClass.getConstructor(String.class);
    return constructor.newInstance(message);
  }

  @Override
  public AuthServiceException parse(String data) throws Exception {
    return null;
  }

  @Override
  public boolean isParseJson() {
    return false;
  }
}
