package org.qas.api.internal.util.key.support;

import org.qas.api.internal.util.Encoders;
import org.qas.api.internal.util.key.UniqueKey;
import org.qas.api.internal.util.key.UniqueKeyException;
import org.qas.api.internal.util.key.UniqueKeyFactory;
import org.qas.api.internal.util.key.UniqueKeyImportException;

import java.net.NetworkInterface;
import java.util.Enumeration;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * MachineKeyFactory
 *
 * @author Dzung Nguyen
 * @version $Id MachineKeyFactory 2014-03-27 14:12:30z dungvnguyen $
 * @since 1.0
 */
public final class MachineKeyFactory extends UniqueKeyFactory {
  //~ class static properties =================================================
  private static final Logger LOG = Logger.getLogger(MachineKeyFactory.class.getName());
  private static final int machine;
  private static AtomicInteger nextInc = new AtomicInteger(new Random().nextInt());

  static {
    try {
      // build machine piece.
      int machinePiece;
      {
        try {
          StringBuilder builder = new StringBuilder();
          Enumeration<NetworkInterface> enw = NetworkInterface.getNetworkInterfaces();
          while (enw.hasMoreElements()) {
            builder.append(enw.nextElement().toString());
          }
          machinePiece = builder.toString().hashCode() << 16;
        } catch (Throwable ex) {
          LOG.log(Level.WARNING, ex.getMessage(), ex);
          machinePiece = (new Random().nextInt()) << 16;
        }

        if (LOG.isLoggable(Level.ALL)) {
          LOG.log(Level.ALL, "machine piece post: " + Integer.toHexString(machinePiece));
        }
      }

      // build process piece.
      final int processPiece;
      {
        int processId = new Random().nextInt();
        try {
          processId = java.lang.management.ManagementFactory.getRuntimeMXBean().getName().hashCode();
        } catch (Throwable ex) {
          LOG.log(Level.WARNING, ex.getMessage(), ex);
        }

        ClassLoader loader = MachineKeyFactory.class.getClassLoader();
        int loaderId = (loader != null ? System.identityHashCode(loader) : 0);

        StringBuilder builder = new StringBuilder();
        builder.append(Integer.toHexString(processId));
        builder.append(Integer.toHexString(loaderId));
        processPiece = builder.toString().hashCode() & 0xFFFF;
        if (LOG.isLoggable(Level.ALL)) {
          LOG.log(Level.ALL, "process pieces: " + Integer.toHexString(processPiece));
        }
      }

      machine = machinePiece | processPiece;
      if (LOG.isLoggable(Level.ALL)) {
        LOG.log(Level.ALL, "machine: " + Integer.toHexString(machine));
      }
    } catch (Exception ex) {
      throw new RuntimeException(ex);
    }
  }

  //~ class members ===========================================================
  @Override
  public UniqueKey generate() {
    return new MachineKey((int) (System.currentTimeMillis() / 1000), machine, nextInc.getAndIncrement());
  }

  @Override
  public UniqueKey from(String source) throws UniqueKeyException {
    if (source == null || source.length() != 14) {
      throw new UniqueKeyImportException("The key {" + source + "} is invalid.");
    }

    return MachineKey.from(Encoders.hexToBytes(source));
  }

  @Override
  public UniqueKey from(byte[] source) throws UniqueKeyException {
    return MachineKey.from(source);
  }
}
