/*
 * Copyright (C) 2011 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */
package org.qas.api.internal.util.google.hash;

/**
 * Static methods to obtain {@link HashFunction} instances, and other static hashing-related
 * utilities.
 *
 * <p>A comparison of the various hash functions can be found
 * <a href="http://goo.gl/jS7HH">here</a>.
 *
 * @author Kevin Bourrillion
 * @author Dimitris Andreou
 * @author Kurt Alfred Kluever
 * @author Dzung Nguyen
 * @version $Id Hashing 2014-03-27 15:03:30z dungvnguyen $
 * @since 1.0
 */
public final class Hashing {
  //~ class members ===========================================================
  /**
   * Returns a hash function implementing the MD5 hash algorithm (128 hash bits) by delegating to
   * the MD5 {@link java.security.MessageDigest}.
   */
  public static HashFunction md5() {
    return Md5Holder.MD5;
  }

  private static class Md5Holder {
    static final HashFunction MD5 = new MessageDigestHashFunction("MD5", "Hashing.md5()");
  }

  /**
   * Returns a hash function implementing the SHA-1 algorithm (160 hash bits) by delegating to the
   * SHA-1 {@link java.security.MessageDigest}.
   */
  public static HashFunction sha1() {
    return Sha1Holder.SHA_1;
  }

  private static class Sha1Holder {
    static final HashFunction SHA_1 =
        new MessageDigestHashFunction("SHA-1", "Hashing.sha1()");
  }

  /**
   * Returns a hash function implementing the SHA-256 algorithm (256 hash bits) by delegating to
   * the SHA-256 {@link java.security.MessageDigest}.
   */
  public static HashFunction sha256() {
    return Sha256Holder.SHA_256;
  }

  private static class Sha256Holder {
    static final HashFunction SHA_256 =
        new MessageDigestHashFunction("SHA-256", "Hashing.sha256()");
  }

  /**
   * Returns a hash function implementing the SHA-512 algorithm (512 hash bits) by delegating to the
   * SHA-512 {@link java.security.MessageDigest}.
   */
  public static HashFunction sha512() {
    return Sha512Holder.SHA_512;
  }

  private static class Sha512Holder {
    static final HashFunction SHA_512 =
        new MessageDigestHashFunction("SHA-512", "Hashing.sha512()");
  }
}
