package org.qas.api.internal.util;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.CRC32;

/**
 * Crc32ChecksumCalculatingInputStream
 *
 * @author Dzung Nguyen
 * @version $Id Crc32ChecksumCalculatingInputStream 2014-03-27 10:18:30z dungvnguyen $
 * @since 1.0
 */
public class Crc32ChecksumCalculatingInputStream extends FilterInputStream {
  //~ class properties ========================================================
  /** The CRC32 being calculated by this input stream */
  private final CRC32 crc32;

  //~ class members ===========================================================

  public Crc32ChecksumCalculatingInputStream(InputStream in) {
    super(in);
    crc32 = new CRC32();
  }

  public long getCrc32Checksum() {
    return crc32.getValue();
  }

  /**
   * Resets the wrapped input stream and the CRC32 computation.
   *
   * @see java.io.InputStream#reset()
   */
  @Override
  public synchronized void reset() throws IOException {
    crc32.reset();
    in.reset();
  }

  /**
   * @see java.io.InputStream#read()
   */
  @Override
  public int read() throws IOException {
    int ch = in.read();
    if (ch != -1) {
      crc32.update(ch);
    }
    return ch;
  }

  /**
   * @see java.io.InputStream#read(byte[], int, int)
   */
  @Override
  public int read(byte[] b, int off, int len) throws IOException {
    int result = in.read(b, off, len);
    if (result != -1) {
      crc32.update(b, off, result);
    }
    return result;
  }
}
