package org.qas.api;

import org.qas.api.http.Protocol;
import org.qas.api.internal.PropertyContainer;
import org.qas.api.internal.util.Versions;

/**
 * ClientConfiguration
 *
 * @author: Dzung Nguyen
 * @version: $Id ClientConfiguration 2014-03-25 10:59:30z dungvnguyen $
 * @since 1.0
 */
public class ClientConfiguration extends PropertyContainer implements Cloneable {
  //~ class properties ========================================================
  /**
   * the default timeout for a connection socket.
   */
  public static final int DEFAULT_SOCKET_TIMEOUT = 50 * 1000; // 50 seconds.

  /**
   * the default max connection pool size.
   */
  public static final int DEFAULT_MAX_CONNECTIONS = 50;

  /**
   * The default HTTP user agent headers.
   */
  public static final String DEFAULT_USER_AGENT = Versions.getUserAgent();

  /**
   * The default maximum number of retries for error responses.
   */
  public static final int DEFAULT_MAX_RETRIES = 3;

  /**
   * Create the client configuration.
   */
  public ClientConfiguration() {
    setProperty("user-agent", DEFAULT_USER_AGENT);
    setProperty("max-error-retry", DEFAULT_MAX_RETRIES);
    setProperty("protocol", Protocol.HTTPS);
    setProperty("max-connections", DEFAULT_MAX_CONNECTIONS);
    setProperty("socket-timeout", DEFAULT_SOCKET_TIMEOUT);
    setProperty("connection-timeout", 50 * 1000);
    setProperty("socket-buffer-size-hints", new int[]{0, 0});
    setProperty("proxy-port", -1);
  }

  /**
   * Creates the client configuration from the given another
   * {@link ClientConfiguration client configuration}
   *
   * @param other the given {@link ClientConfiguration client configuration}
   *              to copy data.
   */
  public ClientConfiguration(ClientConfiguration other) {
    // copy data from the other configuration.
    this.setPropertiesFrom(other);
  }

  /**
   * @return the setting protocol (HTTP or HTTPS) to use when connecting
   * to the target service.
   */
  public Protocol getProtocol() {
    return (Protocol) getProperty("protocol");
  }

  /**
   * Sets the protocol (i.e. HTTP or HTTPS) to use when connecting to the
   * target service.
   * <p>
   * The default configuration is to use HTTPS for all requests for
   * increased security.
   * </p>
   *
   * @param protocol the protocol to use when connecting to the target service.
   */
  public void setProtocol(Protocol protocol) {
    setProperty("protocol", protocol);
  }

  /**
   * Sets the protocol (i.e. HTTP or HTTPS) to use when connecting to the
   * target service, and returns the update {@link ClientConfiguration}
   * object so that additional calls may be chained together.
   *
   * @param protocol the protocol to use when connecting to the target service.
   * @return the update {@link ClientConfiguration} object with the new
   * protocol setting.
   */
  public ClientConfiguration withProtocol(Protocol protocol) {
    setProtocol(protocol);
    return this;
  }

  /**
   * @return the maximum number of allowed open HTTP connections.
   */
  public int getMaxConnections() {
    return ((Number) getProperty("max-connections")).intValue();
  }

  /**
   * Sets the maximum number of allowed open HTTP connections.
   *
   * @param maxConnections the maximum number of allowed open HTTP connections.
   */
  public void setMaxConnections(int maxConnections) {
    setProperty("max-connections", maxConnections);
  }

  /**
   * Sets the maximum number of allowed open HTTP connections and return
   * the updated {@link ClientConfiguration} object.
   *
   * @param maxConnections the maximum number of allowed open HTTP connection.
   * @return The updated {@link ClientConfiguration} object with the new max
   * HTTP connections setting.
   */
  public ClientConfiguration withMaxConnections(int maxConnections) {
    setMaxConnections(maxConnections);
    return this;
  }

  /**
   * @return the HTTP user agent header to send with all request.
   */
  public String getUserAgent() {
    return (String) getProperty("user-agent");
  }

  /**
   * Sets the HTTP user agent header to send with all request.
   *
   * @param userAgent The user agent string to use when sending request.
   */
  public void setUserAgent(String userAgent) {
    setProperty("user-agent", userAgent);
  }

  /**
   * Sets the HTTP user agent header to send with all request and return the
   * updated {@link ClientConfiguration} object.
   *
   * @param userAgent the given user agent string to use when sending request.
   * @return the updated {@link ClientConfiguration} object with the new user
   * agent setting.
   */
  public ClientConfiguration withUserAgent(String userAgent) {
    setUserAgent(userAgent);
    return this;
  }

  /**
   * @return the proxy host the client will connect through.
   */
  public String getProxyHost() {
    return (String) getProperty("proxy-host");
  }

  /**
   * Sets the optional proxy host the client will connect through.
   *
   * @param proxyHost the proxy host the client will connect through.
   */
  public void setProxyHost(String proxyHost) {
    setProperty("proxy-host", proxyHost);
  }

  /**
   * Sets the optioanl proxy host the client will connect through and returns
   * the updated {@link ClientConfiguration} object.
   *
   * @param proxyHost the proxy host the client will connect through.
   * @return the updated {@link ClientConfiguration} object with the new proxy
   * host setting.
   */
  public ClientConfiguration withProxyHost(String proxyHost) {
    setProxyHost(proxyHost);
    return this;
  }

  /**
   * @return the proxy port the client will connect through.
   */
  public int getProxyPort() {
    Integer port = (Integer) getProperty("proxy-port");
    return (port == null ? -1 : port.intValue());
  }

  /**
   * Sets the optional proxy port the client will connect through.
   *
   * @param proxyPort the proxy port the client will connect through.
   */
  public void setProxyPort(int proxyPort) {
    setProperty("proxy-port", proxyPort);
  }

  /**
   * Sets the optional proxy port the client will connect through and return
   * the updated {@link ClientConfiguration} object.
   *
   * @param proxyPort the proxy port the client will connect through.
   * @return the updated {@link ClientConfiguration} object with the new proxy
   * port setting.
   */
  public ClientConfiguration withProxyPort(int proxyPort) {
    setProxyPort(proxyPort);
    return this;
  }

  /**
   * @return the proxy username the configured client will use if connecting
   * through a proxy.
   */
  public String getProxyUsername() {
    return (String) getProperty("proxy-uname");
  }

  /**
   * Sets the proxy username to use if connecting through a proxy.
   *
   * @param username the proxy username to use if connecting through a proxy.
   */
  public void setProxyUsername(String username) {
    setProperty("proxy-uname", username);
  }

  /**
   * Sets the optional proxy username if connecting through a proxy and return
   * the updated {@link ClientConfiguration} object.
   *
   * @param username the proxy username to use if connecting though a proxy.
   * @return the updated {@link ClientConfiguration} object with a new proxy
   * username setting.
   */
  public ClientConfiguration withProxyUsername(String username) {
    setProxyUsername(username);
    return this;
  }

  /**
   * @return the proxy password the configure client will use if connecting
   * through a proxy.
   */
  public String getProxyPassword() {
    return (String) getProperty("proxy-passwd");
  }

  /**
   * Sets the optional proxy password if connecting through a proxy.
   *
   * @param password the proxy password to use if connecting through a proxy.
   */
  public void setProxyPassword(String password) {
    setProperty("proxy-passwd", password);
  }

  /**
   * Sets the optional proxy password if connecting through a proxy and return
   * the updated {@link ClientConfiguration} object.
   *
   * @param password the proxy password to use if connecting through a proxy.
   * @return the updated {@link ClientConfiguration} object with a new proxy
   * password setting.
   */
  public ClientConfiguration withProxyPassword(String password) {
    setProxyPassword(password);
    return this;
  }

  /**
   * @return the proxy domain for configure an NTLM proxy.
   */
  public String getProxyDomain() {
    return (String) getProperty("proxy-domain");
  }

  /**
   * Sets the optional Windows domain name for configuration an NTLM proxy.
   *
   * @param proxyDomain the optional Windows domain name for configuring an
   *                    NTLM proxy.
   */
  public void setProxyDomain(String proxyDomain) {
    setProperty("proxy-domain", proxyDomain);
  }

  /**
   * Sets the optional Windows domain nam for configuration an NTLM proxy and
   * return the updated {@link ClientConfiguration} object.
   *
   * @param proxyDomain the optional Windows domain name for configuring an
   *                    NTLM proxy.
   * @return the updated {@link ClientConfiguration} object with a new proxy
   * domain setting.
   */
  public ClientConfiguration withProxyDomain(String proxyDomain) {
    setProxyDomain(proxyDomain);
    return this;
  }

  /**
   * @return the optional Windows workstation name for configuring NTLM proxy
   * support.
   */
  public String getProxyWorkstation() {
    return (String) getProperty("proxy-workstation");
  }

  /**
   * Sets the optional Windows workstation name for configuring NTLM proxy
   * support.
   *
   * @param proxyWorkstation the optional Windows workstation name for configuring
   *                         NTLM proxy support.
   */
  public void setProxyWorkstation(String proxyWorkstation) {
    setProperty("proxy-workstation", proxyWorkstation);
  }

  /**
   * Sets the optional Windows workstation name for configuring NTLM proxy
   * support and returns the updated {@link ClientConfiguration} object.
   *
   * @param proxyWorkstation the optional Windows workstation name for configuring
   *                         NTLM proxy support.
   * @return the updated {@link ClientConfiguration} object with a new proxy
   * workstation setting.
   */
  public ClientConfiguration withProxyWorkstation(String proxyWorkstation) {
    setProxyWorkstation(proxyWorkstation);
    return this;
  }

  /**
   * @return the maximum number of retry attempts for failed retryable requests.
   */
  public int getMaxErrorRetry() {
    return ((Number) getProperty("max-error-retry")).intValue();
  }

  /**
   * Sets the maximum number of retry attempts for failed retryable requests.
   *
   * @param maxErrorRetry the maximum number of retry attempts for failed
   *                      retryable requests.
   */
  public void setMaxErrorRetry(int maxErrorRetry) {
    setProperty("max-error-retry", maxErrorRetry);
  }

  /**
   * Sets the maximum number of retry attempts for failed retryable requests
   * and returns the updated {@link ClientConfiguration} object.
   *
   * @param maxErrorRetry the maximum number of retry attempts for failed
   *                      retryable requests.
   * @return the updated {@link ClientConfiguration} object with a new maximum
   * number of retry attempts for failed retryable requests setting.
   */
  public ClientConfiguration withMaxErrorRetry(int maxErrorRetry) {
    setMaxErrorRetry(maxErrorRetry);
    return this;
  }

  /**
   * @return the amount of time wait (in milliseconds) for data to be
   * transferred over an established, open connection before the
   * connection timeout and is closed.
   */
  public int getSocketTimeout() {
    return ((Number) getProperty("socket-timeout")).intValue();
  }

  /**
   * Sets the amount of time wait (in milliseconds) for data to be transferred
   * over an established, open connection before the connection timeout and is
   * closed.
   *
   * @param socketTimeout The amount of time to wait (in milliseconds) for data
   *                      to be transferred over an established, open
   *                      connection before the connection timeout and is
   *                      closed.
   */
  public void setSocketTimeout(int socketTimeout) {
    setProperty("socket-timeout", socketTimeout);
  }

  /**
   * Sets the amount of time wait (in milliseconds) for data to be transferred
   * over an established, open connection before the connection timeout and is
   * closed and returns the updated {@link ClientConfiguration} object.
   *
   * @param socketTimeout The amount of time to wait (in milliseconds) for data
   *                      to be transferred over an established, open
   *                      connection before the connection timeout and is
   *                      closed.
   * @return the updated {@link ClientConfiguration} object with a new socket
   * timeout setting.
   */
  public ClientConfiguration withSocketTimeout(int socketTimeout) {
    setSocketTimeout(socketTimeout);
    return this;
  }

  /**
   * @return the amount of to wait (in milliseconds) when initially
   * establishing a connection before giving up and timing out.
   */
  public int getConnectionTimeout() {
    return ((Number) getProperty("connection-timeout")).intValue();
  }

  /**
   * Sets the amount of to wait (in milliseconds) when initially establishing a
   * connection before giving up and timing out.
   *
   * @param connectionTimeout The amount of time wait (in milliseconds) when
   *                          initially establishing a connection before giving
   *                          up and timing out.
   */
  public void setConnectionTimeout(int connectionTimeout) {
    setProperty("connection-timeout", connectionTimeout);
  }

  /**
   * Sets the amount of to wait (in milliseconds) when initially establishing a
   * connection before giving up and timing out and returns
   * {@link ClientConfiguration} object.
   *
   * @param connectionTimeout The amount of time wait (in milliseconds) when
   *                          initially establishing a connection before giving
   *                          up and timing out.
   * @return the updated {@link ClientConfiguration} object with a new
   * connection timeout setting.
   */
  public ClientConfiguration withConnectionTimeout(int connectionTimeout) {
    setConnectionTimeout(connectionTimeout);
    return this;
  }

  /**
   * @return the optional size hints (in bytes) for the low level TCP send and
   * receive buffers. This is an advanced option for advanced users who
   * want to tune low level TCP parameters to try squeeze out more
   * performance.
   */
  public int[] getSocketBufferSizeHints() {
    return (int[]) getProperty("socket-buffer-size-hints");
  }

  /**
   * Sets the optional size hints (in bytes) for the low level TCP send and
   * receive buffers. This is an advanced option for advanced users who want
   * to tune low level TCP parameters to try and squeeze out more performance.
   *
   * @param socketSendBufferSizeHint    the size hint (in bytes) for the low level
   *                                    TCP send buffer.
   * @param socketReceiveBufferSizeHint the size hint (in bytes) for the low
   *                                    level TCP receive buffer.
   */
  public void setSocketBufferSizeHints(int socketSendBufferSizeHint,
                                       int socketReceiveBufferSizeHint) {
    setProperty("socket-buffer-size-hints", new int[]{
      socketSendBufferSizeHint,
      socketReceiveBufferSizeHint
    });
  }

  /**
   * Sets the optional size hints (in bytes) for the low level TCP send and
   * receive buffers. This is an advanced option for advanced users who want
   * to tune low level TCP parameters to try and squeeze out more performance
   * and returns the updated {@link ClientConfiguration} object.
   *
   * @param socketSendBufferSizeHint    the size hint (in bytes) for the low level
   *                                    TCP send buffer.
   * @param socketReceiveBufferSizeHint the size hint (in bytes) for the low
   *                                    level TCP receive buffer.
   * @return the updated {@link ClientConfiguration} object with a new socket
   * buffer size hints setting.
   */
  public ClientConfiguration withSocketBufferSizeHints(int socketSendBufferSizeHint,
                                                       int socketReceiveBufferSizeHint) {
    setSocketBufferSizeHints(socketSendBufferSizeHint, socketReceiveBufferSizeHint);
    return this;
  }

  /**
   * Sets the property name and value.
   *
   * @param name  the property name.
   * @param value the property value.
   * @return the current {@link ClientConfiguration}.
   */
  public ClientConfiguration withProperty(String name, Object value) {
    setProperty(name, value);
    return this;
  }

  /**
   * @return a shallow copy of this {@link ClientConfiguration} instance.
   */
  @Override
  public ClientConfiguration clone() {
    ClientConfiguration that = new ClientConfiguration();
    that.setPropertiesFrom(this);
    return that;
  }

  @Override
  public String toString() {
    return JsonMapper.toJson(this);
  }
}
