package com.paystack.android.ui.paymentchannels.mobilemoney.mpesa

import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.lifecycle.viewmodel.compose.LocalViewModelStoreOwner
import androidx.lifecycle.viewmodel.compose.viewModel
import com.paystack.android.ui.R
import com.paystack.android.ui.components.LoadingScreen
import com.paystack.android.ui.paymentchannels.mobilemoney.MobileMoneyViewModelProviderFactory
import com.paystack.android.ui.paymentchannels.mobilemoney.mpesa.inprogress.MpesaProgressUi
import com.paystack.android.ui.paymentchannels.mobilemoney.mpesa.numberform.MpesaNumberFormUi
import com.paystack.android.ui.paymentchannels.mobilemoney.mpesa.offline.MpesaOfflineUi
import com.paystack.android.ui.paymentchannels.mobilemoney.mpesa.offline.PayWithLipaSteps
import com.paystack.android.ui.paymentchannels.mobilemoney.mpesa.paymenterror.MpesaOfflineErrorUi
import com.paystack.android.ui.paymentchannels.mobilemoney.mpesa.paymenterror.MpesaPaymentErrorUi
import com.paystack.androiddesignsystem.PaystackTheme

@Composable
internal fun MpesaScreen(
    viewModelsProviderFactory: MobileMoneyViewModelProviderFactory,
    modifier: Modifier = Modifier,
) {
    val viewModel: MpesaViewModel = viewModel(factory = viewModelsProviderFactory)
    val mpesaPaymentsState by viewModel.mpesaPaymentsState.collectAsState()
    val numberFormUiState by viewModel.numberFormUiState.collectAsState()
    val showPaybillNumberCopyIcon by viewModel.showPaybillNumberCopyIcon.collectAsState()
    val showAccountNumberCopyIcon by viewModel.showAccountNumberCopyIcon.collectAsState()

    CompositionLocalProvider(
        LocalViewModelStoreOwner provides viewModel.mpesaFlowViewModelStoreOwner
    ) {

        when (val state = mpesaPaymentsState) {
            MpesaPaymentState.EnterNumber -> MpesaNumberFormUi(
                state = numberFormUiState,
                onPhoneNumberChanged = viewModel::onPhoneNumberChanged,
                modifier = modifier.padding(horizontal = PaystackTheme.dimensions.spacing.double),
                onPayOnline = viewModel::payOnline,
                onSwitchToOffline = viewModel::payOffline
            )

            is MpesaPaymentState.InProgress -> MpesaProgressUi(
                phoneNumber = state.phoneNumber,
                instruction = state.instruction,
                timeLeftMs = state.timeLeftMs,
                timeAllowedMs = state.requeryDelayMs,
                modifier = modifier
            )

            MpesaPaymentState.VerifyingPayment -> LoadingScreen(
                message = stringResource(R.string.pstk_checking_transaction_status),
                modifier = modifier,
            )

            is MpesaPaymentState.Error -> MpesaPaymentErrorUi(
                message = state.message,
                onSwitchToOffline = viewModel::payOffline,
                onRetrySameNumber = viewModel::retrySameNumber,
                onTryAnotherNumber = viewModel::switchToOnline,
                modifier = modifier
            )

            is MpesaPaymentState.OfflineError -> MpesaOfflineErrorUi(
                message = state.message,
                isStatusRequery = state.isStatusRequery,
                onPaymentCompleted = viewModel::checkOfflinePaymentStatus,
                onSwitchToOnline = viewModel::switchToOnline,
                onRetryOffline = viewModel::retryPayOffline,
                modifier = modifier
            )

            is MpesaPaymentState.Loading -> LoadingScreen(
                message = state.message.orEmpty(),
                modifier = modifier,
            )

            is MpesaPaymentState.ShowOfflineDetails -> MpesaOfflineUi(
                amount = state.amount,
                paybillNumber = state.paybillNumber,
                accountNumber = state.accountNumber,
                showPaybillNumberCopyIcon = showPaybillNumberCopyIcon,
                showAccountNumberCopyIcon = showAccountNumberCopyIcon,
                onShowPayWithLipaSteps = viewModel::showPayWithLipaSteps,
                onPaymentCompleted = viewModel::checkOfflinePaymentStatus,
                onSwitchToOnline = viewModel::switchToOnline,
                onPaybillNumberCopied = viewModel::startCopyPaybillNumberCountdownTimer,
                onAccountNumberCopied = viewModel::startCopyAccountNumberCountdownTimer,
                modifier = modifier.padding(horizontal = PaystackTheme.dimensions.spacing.double)
            )

            is MpesaPaymentState.ShowPayWithLipaSteps -> PayWithLipaSteps(
                onBackClick = viewModel::showOfflineDetails,
                modifier = modifier.padding(horizontal = PaystackTheme.dimensions.spacing.double)
            )
        }
    }
}
