package com.paystack.android.ui.models

import com.paystack.android.core.api.models.MobileMoneyChargeResponse

internal data class MobileMoneyCharge(
    val channelName: String,
    val phone: String? = null,
    val provider: String,
    val transactionId: String,
    val action: Action?,
) {
    sealed class Action {
        data class ShowInstruction(val instruction: String, val requeryDelayMs: Int) : Action()

        object EnterNumber : Action()

        data class ShowOfflineDetails(val paybillNumber: String?, val accountNumber: String?) :
            Action()
    }

    companion object {
        private const val SEC_TO_MS_MULTIPLIER = 1000

        fun from(response: MobileMoneyChargeResponse): MobileMoneyCharge {
            val action = when (response.displayParams.type) {
                MobileMoneyChargeResponse.DisplayType.ENTER_NUMBER -> Action.EnterNumber
                MobileMoneyChargeResponse.DisplayType.POP -> Action.ShowInstruction(
                    instruction = response.displayParams.message,
                    requeryDelayMs = response.displayParams.timer * SEC_TO_MS_MULTIPLIER
                )

                MobileMoneyChargeResponse.DisplayType.MPESA_OFFLINE_PAYMENT_INSTRUCTIONS ->
                    Action.ShowOfflineDetails(
                        paybillNumber = response.displayParams.paybillNumber,
                        accountNumber = response.displayParams.accountNumber
                    )

                MobileMoneyChargeResponse.DisplayType.UNSUPPORTED -> null
            }

            return MobileMoneyCharge(
                channelName = response.channelName,
                phone = response.phone,
                provider = response.provider,
                transactionId = response.transactionId,
                action = action
            )
        }
    }
}
