package com.paystack.android.ui.components.views.inputs.textfield

import androidx.compose.animation.animateColor
import androidx.compose.animation.core.LinearEasing
import androidx.compose.animation.core.animateFloat
import androidx.compose.animation.core.spring
import androidx.compose.animation.core.tween
import androidx.compose.animation.core.updateTransition
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.graphics.Color

@Composable
internal fun StackTextFieldTransition(
    inputState: InputPhase,
    contentColor: @Composable (InputPhase) -> Color,
    showLabel: Boolean,
    content: @Composable (
        labelProgress: Float,
        labelContentColor: Color,
        placeholderOpacity: Float,
    ) -> Unit
) {
    // Transitions from/to InputPhase.Focused are the most critical in the transition below.
    // Unfocused <-> UnfocusedNotEmpty are needed when a single state is used to control
    // multiple text fields.
    val transition = updateTransition(inputState, label = "TextFieldInputState")

    val labelProgress by transition.animateFloat(
        label = "LabelProgress",
        transitionSpec = { tween(durationMillis = StackTextFieldAnimationDuration) }) {
        when (it) {
            InputPhase.Focused -> 1f
            InputPhase.UnfocusedEmpty -> 0f
            InputPhase.Unfocused -> 1f
        }
    }

    val placeholderOpacity by transition.animateFloat(
        label = "PlaceholderOpacity",
        transitionSpec = {
            if (InputPhase.Focused isTransitioningTo InputPhase.UnfocusedEmpty) {
                tween(
                    durationMillis = PlaceholderAnimationDelayOrDuration, easing = LinearEasing
                )
            } else if (InputPhase.UnfocusedEmpty isTransitioningTo InputPhase.Focused ||
                InputPhase.Unfocused isTransitioningTo InputPhase.UnfocusedEmpty
            ) {
                tween(
                    durationMillis = PlaceholderAnimationDuration,
                    delayMillis = PlaceholderAnimationDelayOrDuration,
                    easing = LinearEasing
                )
            } else {
                spring()
            }
        }) {
        when (it) {
            InputPhase.Focused -> 1f
            InputPhase.UnfocusedEmpty -> if (showLabel) 0f else 1f
            InputPhase.Unfocused -> 0f
        }
    }

    val labelContentColor by transition.animateColor(
        transitionSpec = { tween(durationMillis = StackTextFieldAnimationDuration) },
        label = "LabelContentColor",
        targetValueByState = contentColor
    )

    content(
        labelProgress,
        labelContentColor,
        placeholderOpacity,
    )
}

internal const val StackTextFieldAnimationDuration = 150
private const val PlaceholderAnimationDuration = 83
private const val PlaceholderAnimationDelayOrDuration = 67
