package com.paystack.android.ui.paymentchannels.mobilemoney.mpesa.numberform

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.paystack.android.core.api.models.AccessCodeData
import com.paystack.android.ui.data.transaction.TransactionRepository
import com.paystack.android.ui.models.MobileMoneyCharge
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

internal class MpesaNumberFormViewModel(
    private val transactionAccessData: AccessCodeData,
    private val providerKey: String,
    private val transactionRepository: TransactionRepository,
    private val resultDispatcher: CoroutineDispatcher = Dispatchers.Main
) : ViewModel() {
    private val _uiState = MutableStateFlow(
        MpesaNumberFormState(
            providerKey = providerKey,
            providers = transactionAccessData.channelOptions.mobileMoney.orEmpty(),
            amount = transactionAccessData.amount,
            currency = transactionAccessData.currency
        )
    )
    val uiState = _uiState.asStateFlow()

    private val _result = MutableStateFlow<Result<MobileMoneyCharge>?>(null)
    val result = _result.asStateFlow()

    fun onPhoneNumberChanged(phoneNumber: String) {
        _uiState.update { it.copy(phoneNumber = phoneNumber) }
    }

    fun submitForm() {
        viewModelScope.launch(resultDispatcher) {
            val formattedPhoneNumber = formatPhoneNumber()
            if (isFormattedPhoneNumberValid(formattedPhoneNumber)) return@launch

            _uiState.update { it.copy(isProcessing = true) }
            _result.value = transactionRepository.chargeMobileMoney(
                transactionId = transactionAccessData.id,
                provider = providerKey,
                phone = formattedPhoneNumber
            )
            _uiState.update { it.copy(isProcessing = false) }
        }
    }

    /**
     * Formats an inputted and validated KE phone number to start with dial code of "+254"
     * and followed by other 9-digits.
     */
    private fun formatPhoneNumber(): String {
        val prefixLen = _uiState.value.phoneNumber.length - KE_STANDARD_PHONE_NUMBER_SUFFIX_LENGTH
        val standardFormatPrefix = _uiState.value.phoneNumber.take(prefixLen)
        return if (standardFormatPrefix != KE_DIALING_CODE) {
            _uiState.value.phoneNumber
                .replaceRange(
                    range = IntRange(start = 0, endInclusive = standardFormatPrefix.length - 1),
                    replacement = KE_DIALING_CODE
                )
        } else _uiState.value.phoneNumber
    }

    /**
     * The standard KE phone number length with dial code is 13.
     */
    private fun isFormattedPhoneNumberValid(formattedPhoneNumber: String): Boolean =
        formattedPhoneNumber.length != KE_STANDARD_PHONE_NUMBER_FULL_LENGTH

    companion object {
        private const val KE_DIALING_CODE = "+254"
        private const val KE_STANDARD_PHONE_NUMBER_SUFFIX_LENGTH = 9
        private const val KE_STANDARD_PHONE_NUMBER_FULL_LENGTH = 13
    }
}
