package com.paystack.android.ui.paymentchannels.mobilemoney.mpesa

import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelStore
import androidx.lifecycle.ViewModelStoreOwner
import androidx.lifecycle.viewModelScope
import com.paystack.android.core.api.models.PaystackError
import com.paystack.android.core.api.models.TransactionStatus
import com.paystack.android.core.logging.Logger
import com.paystack.android.ui.R
import com.paystack.android.ui.data.transaction.TransactionRepository
import com.paystack.android.ui.models.Charge
import com.paystack.android.ui.models.MobileMoneyCharge
import com.paystack.android.ui.models.MobileMoneyCharge.Action
import com.paystack.android.ui.utilities.CountdownTimer
import com.paystack.android.ui.utilities.StringProvider
import com.paystack.android.ui.utilities.isFatal
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

internal class MpesaViewModel(
    private val accessCode: String,
    private val transactionRepository: TransactionRepository,
    private val onPaymentComplete: (Charge) -> Unit,
    private val onError: (Throwable) -> Unit,
    private val stringProvider: StringProvider,
    private val countdownDispatcher: CoroutineDispatcher = Dispatchers.Default
) : ViewModel() {
    private val _mpesaPaymentsState =
        MutableStateFlow<MpesaPaymentState>(MpesaPaymentState.EnterNumber)
    val mpesaPaymentsState: StateFlow<MpesaPaymentState>
        get() = _mpesaPaymentsState

    /*
     * Provide a ViewModelStoreOwner for the different states of the Mpesa Payment flow.
     * This is needed for the individual screens to have their own ViewModelStore for keeping state
     * between configuration changes.
     */
    private val mpesaFlowViewModelStores = mutableMapOf<String, ViewModelStore>()
    val mpesaFlowViewModelStoreOwner: ViewModelStoreOwner = object : ViewModelStoreOwner {
        override val viewModelStore: ViewModelStore
            get() {
                val key = _mpesaPaymentsState.value.javaClass.canonicalName ?: run {
                    Logger.error("Unable to get name for ${_mpesaPaymentsState.value.javaClass}")
                    return ViewModelStore()
                }

                return mpesaFlowViewModelStores[key] ?: run {
                    val store = ViewModelStore()
                    mpesaFlowViewModelStores[key] = store
                    store
                }
            }
    }

    fun processResult(result: Result<MobileMoneyCharge>) {
        result.onSuccess(this::handleSuccessResult)
            .onFailure(this::handleFailure)
    }

    private fun handleSuccessResult(chargeResult: MobileMoneyCharge) {
        val action = chargeResult.action ?: return
        when (action) {
            Action.EnterNumber -> _mpesaPaymentsState.value = MpesaPaymentState.EnterNumber
            is Action.ShowInstruction -> {
                val requeryDelayMs = action.requeryDelayMs
                _mpesaPaymentsState.value = MpesaPaymentState.InProgress(
                    instruction = action.instruction,
                    phoneNumber = chargeResult.phone,
                    timeLeftMs = requeryDelayMs,
                    requeryDelayMs = requeryDelayMs
                )

                val requeryTimerJob = startRequeryDelayTimer(requeryDelayMs, chargeResult.phone)
                observeTransactionEvent(chargeResult.channelName) { status ->
                    if (status == TransactionStatus.Success) {
                        requeryTimerJob.cancel()
                        checkPaymentStatus(chargeResult.phone)
                    }
                }
            }
        }
    }

    private fun handleFailure(error: Throwable) {
        Logger.error(error, error.message.orEmpty())

        if (error is PaystackError && !error.isFatal) {
            _mpesaPaymentsState.update {
                MpesaPaymentState.Error(message = error.message)
            }
            return
        }

        val message = stringProvider.getString(R.string.pstk_generic_error_msg)
        _mpesaPaymentsState.update { MpesaPaymentState.Error(message) }
        onError(error)
    }

    private fun startRequeryDelayTimer(delayTimeMs: Int, phoneNumber: String): Job {
        return viewModelScope.launch(countdownDispatcher) {
            CountdownTimer().start(delayTimeMs).collectLatest { remainingTimeMs ->
                val state = _mpesaPaymentsState.value
                if (state is MpesaPaymentState.InProgress) {
                    _mpesaPaymentsState.value = state.copy(timeLeftMs = remainingTimeMs)

                    if (remainingTimeMs == 0) {
                        checkPaymentStatus(phoneNumber)
                    }
                }
            }
        }
    }

    private fun checkPaymentStatus(phoneNumber: String) {
        viewModelScope.launch {
            _mpesaPaymentsState.value = MpesaPaymentState.VerifyingPayment
            transactionRepository.checkPendingCharge(accessCode)
                .onFailure(this@MpesaViewModel::handleFailure)
                .onSuccess { charge ->
                    when (charge.status) {
                        TransactionStatus.Success -> onPaymentComplete(charge)
                        else -> {
                            val errorMessage = stringProvider.getString(
                                R.string.pstk_mpesa_unreachable_error,
                                phoneNumber
                            )
                            _mpesaPaymentsState.value = MpesaPaymentState.Error(errorMessage)
                        }
                    }
                }
        }
    }

    private fun observeTransactionEvent(channelName: String, onEvent: (TransactionStatus) -> Unit) {
        viewModelScope.launch {
            val result = transactionRepository.awaitTransactionEvent(channelName)
            result.onFailure(this@MpesaViewModel::handleFailure)
                .onSuccess(onEvent)
        }
    }
}
