package com.paystack.android.ui.paymentchannels.card

import androidx.compose.animation.AnimatedContent
import androidx.compose.animation.ExperimentalAnimationApi
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.lifecycle.viewmodel.compose.LocalViewModelStoreOwner
import androidx.lifecycle.viewmodel.compose.viewModel
import com.paystack.android.core.api.models.AccessCodeData
import com.paystack.android.core.api.models.TransactionStatus
import com.paystack.android.ui.R
import com.paystack.android.ui.components.LoadingScreen
import com.paystack.android.ui.models.Charge
import com.paystack.android.ui.paymentchannels.card.auth.address.AddressForm
import com.paystack.android.ui.paymentchannels.card.auth.otp.OtpAuthForm
import com.paystack.android.ui.paymentchannels.card.auth.phone.PhoneNumberAuthForm
import com.paystack.android.ui.paymentchannels.card.auth.pin.PinForm
import com.paystack.android.ui.paymentchannels.card.auth.redirect.RedirectAuth
import com.paystack.android.ui.paymentchannels.card.cardform.views.CardDetails
import com.paystack.android.ui.paymentchannels.card.dateofbirth.views.DateOfBirthView

@Composable
internal fun CardPaymentUi(
    transactionAccessData: AccessCodeData,
    modifier: Modifier = Modifier,
    onComplete: (Charge) -> Unit,
    onError: (Throwable) -> Unit,
) {
    val context = LocalContext.current
    val viewModelsProviderFactory = CardViewModelsProviderFactory(
        context = context,
        transactionAccessData = transactionAccessData,
        onPaymentComplete = onComplete,
        onError = onError
    )

    val viewModel: CardPaymentViewModel = viewModel(factory = viewModelsProviderFactory)

    val cardPaymentsState by viewModel.cardPaymentsState.collectAsState()

    CompositionLocalProvider(LocalViewModelStoreOwner provides viewModel.cardFlowViewModelStoreOwner) {
        CardPaymentFlow(
            viewModelsProviderFactory = viewModelsProviderFactory,
            cardPaymentsState = cardPaymentsState,
            isTestTransaction = transactionAccessData.isTestTransaction,
            onChargeResult = viewModel::processCardChargeResponse,
            process3dsResult = viewModel::process3dsResponse,
            onRestartPayment = viewModel::restartCardPayment,
            modifier = modifier,
        )
    }
}

@Composable
@OptIn(ExperimentalAnimationApi::class)
private fun CardPaymentFlow(
    viewModelsProviderFactory: CardViewModelsProviderFactory,
    cardPaymentsState: CardPaymentsState,
    isTestTransaction: Boolean,
    onChargeResult: (Result<Charge>) -> Unit,
    process3dsResult: (Result<TransactionStatus>) -> Unit,
    onRestartPayment: () -> Unit,
    modifier: Modifier = Modifier
) {
    AnimatedContent(targetState = cardPaymentsState, label = "CardPaymentHostAnim") { state ->
        when (state) {
            is CardPaymentsState.CardDetails -> CardDetails(
                viewModelsProviderFactory,
                onResult = onChargeResult,
                modifier = modifier,
            )

            is CardPaymentsState.CardPin -> {
                PinForm(
                    viewModelsProviderFactory = viewModelsProviderFactory,
                    modifier = modifier,
                    onComplete = onChargeResult,
                    onCancel = onRestartPayment
                )
            }

            is CardPaymentsState.DateOfBirthAuthentication -> {
                DateOfBirthView(
                    modifier = modifier,
                    onComplete = { },
                    onCancel = onRestartPayment
                )
            }

            is CardPaymentsState.AddressAuthentication -> AddressForm(
                viewModelsProviderFactory = viewModelsProviderFactory,
                countryCode = state.countryCode,
                modifier = modifier,
                onResult = onChargeResult,
                onCancel = onRestartPayment
            )

            CardPaymentsState.PhoneNumberAuth -> PhoneNumberAuthForm(
                modifier = modifier,
                viewModelProviderFactory = viewModelsProviderFactory,
                onResult = onChargeResult,
            )

            is CardPaymentsState.Error -> CardPaymentError(
                errorMessage = state.message,
                isTestTransaction = isTestTransaction,
                onRetryClick = onRestartPayment,
                modifier = modifier
            )

            is CardPaymentsState.OtpAuth -> OtpAuthForm(
                instruction = state.instruction ?: stringResource(R.string.pstk_otp_instruction),
                viewModelsProviderFactory = viewModelsProviderFactory,
                modifier = modifier,
                onResult = onChargeResult,
                onCancel = onRestartPayment
            )

            is CardPaymentsState.RedirectAuth -> {
                RedirectAuth(
                    authUrl = state.authUrl,
                    viewModelsProviderFactory = viewModelsProviderFactory,
                    modifier = modifier,
                    onResult = process3dsResult,
                    onCancelTransaction = onRestartPayment
                )
            }

            CardPaymentsState.Pending -> {
                LoadingScreen(
                    modifier = modifier,
                    message = stringResource(R.string.pstk_checking_transaction_status),
                )
            }
        }
    }
}
