package com.paystack.android.ui.paymentchannels

import com.paystack.android.core.api.models.ChannelOptions
import com.paystack.android.core.api.models.PaymentChannel
import com.paystack.android.ui.R
import com.paystack.android.ui.models.PaymentChannelMenuItem
import com.paystack.android.ui.utilities.StringProvider

/**
 * Builds the list of menu items to display on Payment Channel screen
 * This holds logic that what and how payment channels are shown to users.
 */
internal class PaymentChannelMenuBuilder(
    private val stringProvider: StringProvider,
) {
    fun build(
        channels: List<PaymentChannel>,
        paymentChannelOptions: ChannelOptions,
    ): List<PaymentChannelMenuItem> {
        return channels.flatMap { paymentChannel ->
            createMenuItems(paymentChannel, paymentChannelOptions)
        }
    }

    private fun createMenuItems(
        paymentChannel: PaymentChannel,
        paymentChannelOptions: ChannelOptions
    ) = when (paymentChannel) {
        PaymentChannel.CARD -> listOf(
            PaymentChannelMenuItem(
                paymentChannel,
                paymentChannel.name,
                stringProvider.getString(R.string.pstk_card),
                R.drawable.pstk_ic_channel_card
            )
        )

        PaymentChannel.MOBILE_MONEY -> paymentChannelOptions.mobileMoney.orEmpty()
            .filter { it.key.uppercase() in supportedMobileMoneyOptions }
            .map {
                val iconRes = R.drawable.pstk_ic_channel_mpesa
                PaymentChannelMenuItem(paymentChannel, it.key, it.value, iconRes)
            }

        PaymentChannel.UNKNOWN -> emptyList()
    }

    companion object {
        internal const val KEY_MPESA = "MPESA"
        private val supportedMobileMoneyOptions = listOf(KEY_MPESA)
    }
}
