package com.paystack.android.ui

import android.os.Parcelable
import com.paystack.android.core.api.models.AccessCodeData
import com.paystack.android.core.api.models.PaystackError
import com.paystack.android.ui.models.Charge
import com.paystack.android.ui.utilities.isFatal
import kotlinx.parcelize.Parcelize

/**
 * The state of the payments process. This is driven by responses from Paystack's API
 * The different states here should hold the data needed to display its UI.
 */
internal sealed class PaymentContainerState : Parcelable {
    @Parcelize
    object Initialization : PaymentContainerState()

    @Parcelize
    data class PaymentProcess(val accessCodeData: AccessCodeData) : PaymentContainerState()

    @Parcelize
    data class Success(
        val charge: Charge,
        val merchantName: String,
    ) : PaymentContainerState() {
        override val isTerminal: Boolean
            get() = true
    }

    @Parcelize
    data class Error(val displayText: String, val error: Throwable) : PaymentContainerState() {
        constructor(error: Throwable) : this(error.message.orEmpty(), error)

        override val isTerminal: Boolean
            get() = if (error is PaystackError) {
                error.isFatal
            } else {
                true
            }
    }

    open val isTerminal: Boolean
        get() = false
}
