package com.paystack.android.ui.data.transaction

import com.paystack.android.core.api.await
import com.paystack.android.core.api.models.AccessCodeData
import com.paystack.android.core.api.models.AddressState
import com.paystack.android.core.api.models.CardParams
import com.paystack.android.core.api.models.PaymentChannel
import com.paystack.android.core.api.models.RedirectAuthResponse
import com.paystack.android.core.api.models.TransactionEvent
import com.paystack.android.core.api.models.TransactionStatus
import com.paystack.android.core.api.services.cards.CardPaymentsService
import com.paystack.android.core.api.services.cards.RedirectAuthResultListener
import com.paystack.android.core.api.services.mobilemoney.MobileMoneyService
import com.paystack.android.core.api.services.transactions.PaystackTransactionService
import com.paystack.android.core.api.services.transactions.TransactionEventListener
import com.paystack.android.ui.models.Charge
import com.paystack.android.ui.models.MobileMoneyCharge
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

internal class TransactionRepositoryImpl(
    private val paystackTransactionService: PaystackTransactionService,
    private val cardPaymentsService: CardPaymentsService,
    private val mobileMoneyService: MobileMoneyService,
) : TransactionRepository {
    override suspend fun getByAccessCode(accessCode: String): Result<AccessCodeData> {
        return runCatching {
            paystackTransactionService.verifyAccessCode(accessCode).await()
        }
    }

    override suspend fun chargeCard(
        accessCode: String,
        cardParams: CardParams,
        encryptionKey: String,
    ): Result<Charge> {
        return runCatching {
            val response =
                cardPaymentsService.charge(accessCode, cardParams, encryptionKey).await()
            Charge.from(response)
        }
    }

    override suspend fun chargeMobileMoney(
        transactionId: String,
        provider: String,
        phone: String,
    ): Result<MobileMoneyCharge> {
        return runCatching {
            val response = mobileMoneyService.charge(transactionId, provider, phone)
                .await()
            MobileMoneyCharge.from(response)
        }
    }

    override suspend fun checkPendingCharge(accessCode: String): Result<Charge> {
        return runCatching {
            val response = paystackTransactionService.checkPendingCharge(accessCode).await()
            Charge.from(response)
        }
    }

    override suspend fun authenticateWithPin(
        accessCode: String,
        pin: String,
        encryptionKey: String
    ): Result<Charge> {
        return runCatching {
            val response = cardPaymentsService.authenticateWithPin(accessCode, pin, encryptionKey)
                .await()
            Charge.from(response)
        }
    }

    override suspend fun getAddressStateList(countryCode: String): Result<List<AddressState>> {
        return runCatching {
            cardPaymentsService.getAddressStateList(countryCode).await()
        }
    }

    override suspend fun authenticateWithAddress(
        accessCode: String,
        address: String,
        city: String,
        state: String,
        zipCode: String,
    ): Result<Charge> {
        return runCatching {
            val response = cardPaymentsService.authenticateWithAddress(
                accessCode = accessCode,
                address = address,
                city = city,
                state = state,
                zipCode = zipCode
            ).await()
            Charge.from(response)
        }
    }

    override suspend fun authenticateWithOtp(accessCode: String, otp: String): Result<Charge> {
        return runCatching {
            val response = cardPaymentsService.authenticateWithOtp(accessCode, otp).await()
            Charge.from(response)
        }
    }

    override suspend fun authenticateWithPhone(
        accessCode: String,
        phoneNumber: String
    ): Result<Charge> {
        return runCatching {
            val response = cardPaymentsService.authenticateWithPhone(accessCode, phoneNumber)
                .await()
            Charge.from(response)
        }
    }

    override suspend fun authenticateWithBirthday(
        accessCode: String,
        day: String,
        month: String,
        year: String
    ): Result<Charge> {
        return runCatching {
            val response =
                cardPaymentsService.authenticateWithBirthday(accessCode, day, month, year)
                    .await()
            Charge.from(response)
        }
    }

    override suspend fun await3dsResult(transactionId: String): Result<TransactionStatus> {
        return suspendCoroutine { continuation ->
            val callback = object : RedirectAuthResultListener {
                override fun onResult(response: RedirectAuthResponse) {
                    continuation.resume(Result.success(response.status))
                }

                override fun onError(exception: Throwable) {
                    continuation.resume(Result.failure(exception))
                }
            }
            cardPaymentsService.listenFor3dsResult(transactionId, callback)
        }
    }

    override fun getSupportedPaymentChannels(): List<PaymentChannel> {
        return paystackTransactionService.getSupportedPaymentChannels()
    }

    override suspend fun awaitTransactionEvent(channelName: String): Result<TransactionStatus> {
        return suspendCoroutine { continuation ->
            val callback = object : TransactionEventListener {
                override fun onResult(eventData: TransactionEvent) {
                    continuation.resume(Result.success(eventData.status))
                }

                override fun onError(exception: Throwable) {
                    continuation.resume(Result.failure(exception))
                }
            }
            paystackTransactionService.listenForTransactionEvent(channelName, callback)
        }
    }
}
