package com.paystack.android_sdk.ui.paymentchannels.card.auth.address

import androidx.compose.animation.Crossfade
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.lifecycle.viewmodel.compose.viewModel
import com.paystack.android_sdk.ui.R
import com.paystack.android_sdk.ui.components.buttons.PrimaryButton
import com.paystack.android_sdk.ui.components.buttons.StackTextButton
import com.paystack.android_sdk.ui.components.progressbars.LoadingIndicator
import com.paystack.android_sdk.ui.components.views.inputs.PrimaryDropDownMenu
import com.paystack.android_sdk.ui.components.views.inputs.PrimaryTextInputField
import com.paystack.android_sdk.ui.components.views.text.PaymentTitleText
import com.paystack.android_sdk.ui.models.Charge
import com.paystack.android_sdk.ui.paymentchannels.card.CardViewModelsProviderFactory
import com.paystack.androiddesignsystem.PaystackTheme

@Composable
internal fun AddressForm(
    viewModelsProviderFactory: CardViewModelsProviderFactory,
    countryCode: String,
    onResult: (Result<Charge>) -> Unit,
    onCancel: () -> Unit,
    modifier: Modifier = Modifier,
    viewModel: AddressFormViewModel = viewModel(factory = viewModelsProviderFactory),
) {
    val state by viewModel.state.collectAsState()
    val result by viewModel.result.collectAsState()

    LaunchedEffect(countryCode) {
        viewModel.loadStateList(countryCode)
    }

    LaunchedEffect(result) {
        result?.let(onResult)
    }

    Column(
        modifier = modifier,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {

        PaymentTitleText(
            titleText = stringResource(id = R.string.pstk_card_payment_address_title),
            modifier = Modifier.padding(horizontal = PaystackTheme.dimensions.spacing.triple)
        )

        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

        Crossfade(
            targetState = state.isSubmitting,
            label = "DateOfBirthAnim"
        ) { showLoadingIndicator ->
            Column(horizontalAlignment = Alignment.CenterHorizontally) {
                if (showLoadingIndicator) {
                    Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.triple))

                    LoadingIndicator(
                        modifier = Modifier.size(PaystackTheme.dimensions.spacing.quad),
                        color = PaystackTheme.colors.primaryColors.navy04
                    )
                } else {
                    PrimaryTextInputField(
                        modifier = Modifier.fillMaxWidth(),
                        text = state.streetAddress,
                        onTextChanged = viewModel::setStreetAddress,
                        label = stringResource(id = R.string.pstk_card_payment_address_street_label),
                        placeholder = stringResource(id = R.string.pstk_card_payment_address_street_placeholder),
                    )

                    Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

                    PrimaryTextInputField(
                        modifier = Modifier.fillMaxWidth(),
                        text = state.zipCode,
                        keyboardType = KeyboardType.Number,
                        maxLength = ZIP_CODE_CHARACTER_LIMIT,
                        onTextChanged = viewModel::setZipCode,
                        label = stringResource(id = R.string.pstk_card_payment_address_zipcode_label),
                        placeholder = stringResource(id = R.string.pstk_card_payment_address_zipcode_placeholder)
                    )

                    Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        horizontalArrangement = Arrangement
                            .spacedBy(PaystackTheme.dimensions.spacing.double)
                    ) {
                        PrimaryDropDownMenu(
                            modifier = Modifier.weight(1f),
                            menuItems = state.stateList.map { it.name },
                            label = stringResource(id = R.string.pstk_card_payment_address_state_label),
                            selectedItemIndex = state.selectedStateIndex,
                            onItemSelected = { index ->
                                viewModel.setSelectedState(index)
                            }
                        )

                        PrimaryTextInputField(
                            modifier = Modifier.weight(1f),
                            text = state.city,
                            onTextChanged = viewModel::setCity,
                            label = stringResource(id = R.string.pstk_card_payment_address_city_label),
                            placeholder = stringResource(id = R.string.pstk_card_payment_address_city_placeholder)
                        )
                    }

                    Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

                    PrimaryButton(
                        buttonText = stringResource(id = R.string.pstk_component_pay_button_complete_payment),
                        isEnabled = state.isFormValid,
                        onClick = viewModel::submit,
                        isLoading = state.isSubmitting
                    )

                    StackTextButton(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(PaystackTheme.dimensions.spacing.double),
                        text = stringResource(id = R.string.pstk_card_payment_cancel_transaction),
                        onClick = onCancel
                    )
                }
            }
        }
    }
}

private const val ZIP_CODE_CHARACTER_LIMIT = 10
