package com.paystack.android_sdk.ui.utilities

import java.math.BigDecimal
import java.text.DecimalFormat

internal object CurrencyFormatter {
    private const val MAX_FRACTION_DIGITS = 2

    /**
     * Format amount value as for display.
     * @param amount the value to be formatted
     * @param showEmptyFractions Option to show/hide trailing empty decimal parts of the value.
     * @param showCommaSeparators Option to show/hide separators. e.g. 1,000 vs 1000
     * @return [String] of value formatted for display
     */
    fun format(
        amount: Long,
        currencyCode: String,
        showEmptyFractions: Boolean = false,
        showCommaSeparators: Boolean = true,
    ): String {
        val formattedAmount = moneyFormat(amount, showEmptyFractions, showCommaSeparators)
        return "$currencyCode $formattedAmount"
    }

    /**
     * Format amount value as for display.
     * @param amount the value to be formatted
     * @param showEmptyFractions Option to show/hide trailing empty decimal parts of the value.
     * @param showCommaSeparators Option to show/hide separators. e.g. 1,000 vs 1000
     * @return [String] of value formatted for display
     */
    private fun moneyFormat(
        amount: Long,
        showEmptyFractions: Boolean = false,
        showCommaSeparators: Boolean = true
    ): String {
        val value = BigDecimal(amount)
            .abs()
            .divide(BigDecimal("100"))
            .setScale(2)

        val formatPattern = if (showCommaSeparators) "###,###.##" else "######.##"
        val formatter = DecimalFormat(formatPattern)

        formatter.minimumFractionDigits = if (showEmptyFractions) {
            MAX_FRACTION_DIGITS
        } else {
            getMinimumDecimalDigits(value)
        }
        formatter.maximumFractionDigits = MAX_FRACTION_DIGITS
        val symbols = formatter.decimalFormatSymbols
        symbols.groupingSeparator = ','
        formatter.decimalFormatSymbols = symbols
        return formatter.format(value.toDouble())
    }

    /**
     * Returns the minimum number of digits that should be in a formatted amount.
     * Amounts should be formatted with 2 decimal digits if the decimal part is non-zero.
     */
    private fun getMinimumDecimalDigits(amount: BigDecimal): Int {
        val remainder = amount.remainder(BigDecimal.ONE)
        return if (remainder > BigDecimal.ZERO) {
            2
        } else {
            0
        }
    }
}
