package com.paystack.android_sdk.ui.paymentchannels.card.auth.otp

import androidx.annotation.VisibleForTesting
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.paystack.android_sdk.core.api.models.AccessCodeData
import com.paystack.android_sdk.ui.data.transaction.TransactionRepository
import com.paystack.android_sdk.ui.models.Charge
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

internal class OtpFormViewModel(
    private val transactionAccessData: AccessCodeData,
    private val transactionRepository: TransactionRepository,
    private val otpCountdownTimer: OtpCountdownTimer
) : ViewModel() {
    private val _state = MutableStateFlow(OtpAuthState())
    val state: StateFlow<OtpAuthState> = _state.asStateFlow()

    private val _result = MutableStateFlow<Result<Charge>?>(null)
    val result = _result.asStateFlow()

    init {
        startResendDelayCountdown()
    }

    fun onOtpChanged(otp: String) {
        _state.update { it.copy(otp = otp) }
    }

    fun requestResendOtp() {
        viewModelScope.launch {
            _state.update { it.copy(otpResendAttempts = it.otpResendAttempts.inc()) }
            startResendDelayCountdown()
        }
    }

    fun submit() {
        viewModelScope.launch {
            _state.update { it.copy(isSubmitting = true) }
            _result.value = transactionRepository.authenticateWithOtp(
                accessCode = transactionAccessData.accessCode,
                otp = _state.value.otp
            )
            _state.update { it.copy(isSubmitting = false) }
        }
    }

    @VisibleForTesting(otherwise = VisibleForTesting.PROTECTED)
    public override fun onCleared() {
        super.onCleared()
        otpCountdownTimer.clear()
    }

    private fun startResendDelayCountdown() {
        otpCountdownTimer.start { timeMillis ->
            _state.update { it.copy(otpResendDelayMillis = timeMillis) }
        }
    }
}
