package com.paystack.android_sdk.ui.models

import android.os.Parcelable
import com.paystack.android_sdk.core.api.models.TransactionStatus
import kotlinx.parcelize.Parcelize

@Parcelize
data class Charge(
    val status: TransactionStatus,

    /**
     * The transaction amount expressed as a positive integer in the smallest currency unit.
     * (e.g. 10000 Kobo to represent ₦100 or 100 for $1)
     */
    val amount: Long,

    /**
     * Three-letter ISO currency code.
     * [Supported Currencies](https://paystack.com/api/#supported-currency)
     */
    val currency: String,

    /**
     * Unique transaction reference. Only -, ., = and alphanumeric characters allowed.
     */
    val reference: String,

    /**
     * A URL to visit to authenticate transactions that require external validation.
     * This is available when [status] is [TransactionStatus.OpenUrl].
     * This is primarily used for 3DS transaction authentication
     */
    val authUrl: String?,

    /**
     * Message to display for certain transaction states.
     * This may be available when [TransactionStatus] requires user action.
     */
    val displayText: String?,

    /**
     * Transaction Error messages
     */
    val message: String?,

    /**
     * Country code used for Address verificatuib
     * This may be available when [status] is [TransactionStatus.SendAddress].
     */
    val countryCode: String?,
) : Parcelable {
    companion object {
        fun from(chargeResponse: com.paystack.android_sdk.core.api.models.ChargeResponse) = Charge(
            status = chargeResponse.status,
            amount = chargeResponse.amount,
            currency = chargeResponse.currency,
            authUrl = chargeResponse.authUrl,
            displayText = chargeResponse.displayText,
            message = chargeResponse.message,
            reference = chargeResponse.reference,
            countryCode = chargeResponse.authorization?.countryCode,
        )
    }
}
