package com.paystack.android_sdk.ui.components.views.inputs.validators

import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.input.OffsetMapping
import androidx.compose.ui.text.input.TransformedText

/**
 * Singleton that formats the card according to the provider scheme.
 * This singleton is designed to work within a composable, as we need to provide an Annotated
 * String as a parameter to the single exposed method
 */
internal object CardNumberFormatter {

    /**
     * Method that, given a card type and the card number, we return the transformed text
     * with the required offsets.
     *
     * @param cardType - The identified card scheme/provider of the given card number
     * @param cardNumber - the provided card number entered into the user interface
     * @return The transformed text object that contains the formatted text in Annotated String
     * format, as well as the offset translator
     *
     */
    fun formatCardNumberByType(cardType: CardType, cardNumber: AnnotatedString): TransformedText {
        val number = cardNumber.text
        return when (cardType) {
            CardType.AMEX -> formatAmexCardNumbers(cardNumber = number)
            CardType.VERVE -> formatVerveCardNumbers(cardNumber = number)
            else -> formatOtherCreditCards(cardNumber = number)
        }
    }

    /**
     * Function that matches cards provided by American Express.  These numbers usually
     * start with 34 or 37 For example:
     *
     * Original number: 341234567890123
     * Formatted Number: 3412 345678 90123
     *
     * @param cardNumber
     * @return card number formatted in the American Express format
     */
    private fun formatAmexCardNumbers(cardNumber: String): TransformedText {
        val trimmedAmexCardNumber = if (cardNumber.length > 15) {
            cardNumber.substring(0..14)
        } else {
            cardNumber
        }

        var output = ""

        trimmedAmexCardNumber.indices.forEach { index ->
            output += trimmedAmexCardNumber[index]

            if (index != 14) {
                if (index == 3 || index == 9) {
                    output += " "
                }
            }
        }

        val amexCardOffsetTranslator = object : OffsetMapping {
            override fun originalToTransformed(offset: Int): Int {
                if (offset <= 3) return offset
                if (offset <= 9) return offset + 1
                if (offset <= 15) return offset + 2
                return 17
            }

            override fun transformedToOriginal(offset: Int): Int {
                if (offset <= 4) return offset
                if (offset <= 9) return offset - 1
                if (offset <= 17) return offset - 2
                return 15
            }
        }
        return TransformedText(AnnotatedString(output), amexCardOffsetTranslator)
    }

    private fun formatVerveCardNumbers(cardNumber: String): TransformedText {
        val trimmedVerveCardNumber = if (cardNumber.length == 19) {
            cardNumber.substring(0..18)
        } else {
            cardNumber
        }

        var output = ""

        trimmedVerveCardNumber.indices.forEach { index ->
            output += trimmedVerveCardNumber[index]

            if (index != 18) {
                if (index % 5 == 4) {
                    output += " "
                }
            }
        }

        val verveCardOffsetTranslator = object : OffsetMapping {
            override fun originalToTransformed(offset: Int): Int {
                if (offset <= 4) return offset
                if (offset <= 9) return offset + 1
                if (offset <= 14) return offset + 2
                if (offset <= 19) return offset + 3
                return 23
            }

            override fun transformedToOriginal(offset: Int): Int {
                if (offset <= 5) return offset
                if (offset <= 11) return offset - 1
                if (offset <= 17) return offset - 2
                if (offset <= 23) return offset - 3
                return 19
            }
        }
        return TransformedText(AnnotatedString(output), verveCardOffsetTranslator)
    }

    /**
     * Function that matches cards provided by Mastercard, Maestro, Verve and Visa cards.
     * These cards generally hold the same card number and separation structure. For example:
     *
     * Original card number: number: 5123456789012346
     * Formatted Number: 5123 4567 8901 2346
     *
     * @param cardNumber
     * @return card number formatted in Mastercard, Maestro and Visa card formats.
     */
    private fun formatOtherCreditCards(cardNumber: String): TransformedText {
        val trimmedCardNumber = if (cardNumber.length > 19) {
            cardNumber.substring(0..18)
        } else {
            cardNumber
        }

        var output = ""
        val lastValidIndex = trimmedCardNumber.length - 1

        trimmedCardNumber.indices.forEach { index ->
            output += trimmedCardNumber[index]
            if (index % 4 == 3 && index != lastValidIndex) {
                output += " "
            }
        }

        val creditCardOffsetTranslator = object : OffsetMapping {
            override fun originalToTransformed(offset: Int): Int {
                return if (offset >= trimmedCardNumber.length) {
                    output.length
                } else {
                    offset + offset / 4
                }
            }

            override fun transformedToOriginal(offset: Int): Int {
                return if (offset >= output.length) {
                    trimmedCardNumber.length
                } else {
                    offset - offset / 5
                }
            }
        }
        return TransformedText(AnnotatedString(output), creditCardOffsetTranslator)
    }
}
