package com.paystack.android.ui.data.transaction

import com.paystack.android.core.api.models.AccessCodeData
import com.paystack.android.core.api.models.AddressState
import com.paystack.android.core.api.models.CardParams
import com.paystack.android.core.api.models.PaymentChannel
import com.paystack.android.core.api.models.TransactionEvent
import com.paystack.android.core.api.models.TransactionStatus
import com.paystack.android.ui.models.Charge
import com.paystack.android.ui.models.MobileMoneyCharge

internal interface TransactionRepository {
    suspend fun getByAccessCode(accessCode: String): Result<AccessCodeData>

    /**
     * Attempts a card charge for a transaction with [accessCode].
     * The [encryptionKey] is used to encrypt [cardParams] before the charge attempt.
     *
     * @return a [Result] for [Charge]
     */
    suspend fun chargeCard(
        accessCode: String,
        cardParams: CardParams,
        encryptionKey: String
    ): Result<Charge>

    /**
     * Begin mobile money charge process.
     *
     * @return a [Result] for [Charge]
     */
    suspend fun chargeMobileMoney(
        transactionId: String,
        provider: String,
        phone: String? = null,
    ): Result<MobileMoneyCharge>

    /**
     * Gets the Charge state for a pending transaction.
     *
     * @return a [Result] for [Charge]
     */
    suspend fun checkPendingCharge(accessCode: String): Result<Charge>

    /**
     * Attempts to authenticate transaction with [accessCode] using an address
     *
     * @return a [Result] for [Charge]
     */
    suspend fun authenticateWithAddress(
        accessCode: String,
        address: String,
        city: String,
        state: String,
        zipCode: String,
    ): Result<Charge>

    /**
     * Attempts to authenticate transaction with [accessCode] using a card PIN
     * The [encryptionKey] is used to encrypt [pin] before the authentication attempt.
     *
     * @return a [Result] for [Charge]
     */
    suspend fun authenticateWithPin(
        accessCode: String,
        pin: String,
        encryptionKey: String
    ): Result<Charge>

    /**
     * Attempts to authenticate transaction with [accessCode] using an OTP
     *
     * @return a [Result] for [Charge]
     */
    suspend fun authenticateWithOtp(accessCode: String, otp: String): Result<Charge>

    /**
     * Attempts to authenticate transaction with [accessCode] using a phone number
     *
     * @return a [Result] for [Charge]
     */
    suspend fun authenticateWithPhone(
        accessCode: String,
        phoneNumber: String
    ): Result<Charge>

    /**
     * Attempts to authenticate transaction using a birthday
     * @param accessCode the access code of the transaction
     * @param day the day of the birthday
     * @param month the month of the birthday
     * @param year the year of the birthday
     *
     * @return a [Result] for [Charge]
     */
    suspend fun authenticateWithBirthday(
        accessCode: String,
        day: String,
        month: String,
        year: String
    ): Result<Charge>

    suspend fun getAddressStateList(countryCode: String): Result<List<AddressState>>

    /**
     * Listens for a 3DS transaction result asynchronously using [transactionId]
     * This will be changed to return a UI Charge model instead of a [TransactionStatus]
     * @param transactionId
     *
     * @return a [Result] for [TransactionStatus]
     */
    suspend fun await3dsResult(transactionId: String): Result<TransactionStatus>

    /**
     * Returns a list of payment channels supported by the SDK
     */
    fun getSupportedPaymentChannels(): List<PaymentChannel>

    /**
     * Listens for a transaction event using [channelName]
     * @param channelName
     *
     * @return a [Result] of [TransactionStatus]
     */
    suspend fun awaitTransactionEvent(channelName: String): Result<TransactionEvent>
}
