/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.code;

import com.oracle.svm.core.MemoryWalker;
import com.oracle.svm.core.annotate.Uninterruptible;
import com.oracle.svm.core.code.CodeInfoTable;
import com.oracle.svm.core.code.InstalledCodeObserverSupport;
import com.oracle.svm.core.code.RuntimeMethodInfo;
import com.oracle.svm.core.deopt.Deoptimizer;
import com.oracle.svm.core.deopt.SubstrateInstalledCode;
import com.oracle.svm.core.heap.Heap;
import com.oracle.svm.core.heap.PinnedAllocator;
import com.oracle.svm.core.log.Log;
import com.oracle.svm.core.log.StringBuilderLog;
import com.oracle.svm.core.option.RuntimeOptionKey;
import com.oracle.svm.core.thread.VMOperation;
import com.oracle.svm.core.util.Counter;
import com.oracle.svm.core.util.RingBuffer;
import java.util.Arrays;
import org.graalvm.compiler.options.Option;
import org.graalvm.nativeimage.ImageSingletons;
import org.graalvm.nativeimage.Platform;
import org.graalvm.nativeimage.Platforms;
import org.graalvm.nativeimage.c.function.CodePointer;
import org.graalvm.word.UnsignedWord;
import org.graalvm.word.WordBase;

public class RuntimeCodeInfo {
    private final RingBuffer<String> recentCodeCacheOperations = new RingBuffer();
    private long codeCacheOperationSequenceNumber;
    private final Counter.Group counters = new Counter.Group(CodeInfoTable.Options.CodeCacheCounters, "RuntimeCodeInfo");
    private final Counter lookupMethodCount = new Counter(this.counters, "lookupMethod", "");
    private final Counter addMethodCount = new Counter(this.counters, "addMethod", "");
    private final Counter invalidateMethodCount = new Counter(this.counters, "invalidateMethod", "");
    static final String INFO_ADD = "Add";
    static final String INFO_INVALIDATE = "Invalidate";
    private static final int INITIAL_TABLE_SIZE = 100;
    private RuntimeMethodInfo[] methodInfos;
    private int numMethods;
    private PinnedAllocator tablePin;

    @Platforms(value={Platform.HOSTED_ONLY.class})
    public RuntimeCodeInfo() {
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    public final void tearDown() {
        for (int i = 0; i < this.numMethods; ++i) {
            this.methodInfos[i].freeInstalledCode();
        }
    }

    protected RuntimeMethodInfo lookupMethod(CodePointer ip) {
        this.lookupMethodCount.inc();
        return this.lookupMethodUninterruptible(ip);
    }

    @Uninterruptible(reason="methodInfos is accessed without holding a lock, so must not be interrupted by a safepoint that can add/remove code")
    private RuntimeMethodInfo lookupMethodUninterruptible(CodePointer ip) {
        assert (this.verifyTable());
        if (this.numMethods == 0) {
            return null;
        }
        int idx = RuntimeCodeInfo.binarySearch(this.methodInfos, 0, this.numMethods, ip);
        if (idx >= 0) {
            return this.methodInfos[idx];
        }
        int insertionPoint = -idx - 1;
        if (insertionPoint == 0) {
            assert (((UnsignedWord)ip).belowThan((UnsignedWord)this.methodInfos[0].getCodeStart()));
            return null;
        }
        RuntimeMethodInfo methodInfo = this.methodInfos[insertionPoint - 1];
        assert (((UnsignedWord)ip).aboveThan((UnsignedWord)methodInfo.getCodeStart()));
        if (((UnsignedWord)ip).subtract((UnsignedWord)methodInfo.getCodeStart()).aboveOrEqual(methodInfo.getCodeSize())) {
            return null;
        }
        return methodInfo;
    }

    @Uninterruptible(reason="called from uninterruptible code")
    private static int binarySearch(RuntimeMethodInfo[] a, int fromIndex, int toIndex, CodePointer key) {
        int low = fromIndex;
        int high = toIndex - 1;
        while (low <= high) {
            int mid = low + high >>> 1;
            CodePointer midVal = a[mid].getCodeStart();
            if (((UnsignedWord)midVal).belowThan((UnsignedWord)key)) {
                low = mid + 1;
                continue;
            }
            if (((UnsignedWord)midVal).aboveThan((UnsignedWord)key)) {
                high = mid - 1;
                continue;
            }
            return mid;
        }
        return -(low + 1);
    }

    public void addMethod(RuntimeMethodInfo methodInfo) {
        VMOperation.enqueueBlockingSafepoint("AddMethod", () -> {
            InstalledCodeObserverSupport.activateObservers(methodInfo.codeObserverHandles);
            long num = this.logMethodOperation(methodInfo, INFO_ADD);
            this.addMethodOperation(methodInfo);
            this.logMethodOperationEnd(num);
        });
    }

    private void addMethodOperation(RuntimeMethodInfo methodInfo) {
        VMOperation.guaranteeInProgress("Modifying code tables that are used by the GC");
        this.addMethodCount.inc();
        assert (this.verifyTable());
        if (Options.TraceCodeCache.getValue().booleanValue()) {
            Log.log().string("[Add method: ");
            RuntimeCodeInfo.logMethod(Log.log(), methodInfo);
            Log.log().string("]").newline();
        }
        if (this.methodInfos == null || this.numMethods >= this.methodInfos.length) {
            this.enlargeTable();
            assert (this.verifyTable());
        }
        assert (this.numMethods < this.methodInfos.length);
        int idx = RuntimeCodeInfo.binarySearch(this.methodInfos, 0, this.numMethods, methodInfo.getCodeStart());
        assert (idx < 0) : "must not find code already in table";
        int insertionPoint = -idx - 1;
        System.arraycopy(this.methodInfos, insertionPoint, this.methodInfos, insertionPoint + 1, this.numMethods - insertionPoint);
        ++this.numMethods;
        this.methodInfos[insertionPoint] = methodInfo;
        if (Options.TraceCodeCache.getValue().booleanValue()) {
            this.logTable();
        }
        assert (this.verifyTable());
    }

    private void enlargeTable() {
        VMOperation.guaranteeInProgress("Modifying code tables that are used by the GC");
        Object[] oldMethodInfos = this.methodInfos;
        PinnedAllocator oldTablePin = this.tablePin;
        int newTableSize = Math.max(100, this.numMethods * 2);
        this.tablePin = Heap.getHeap().createPinnedAllocator();
        this.tablePin.open();
        RuntimeMethodInfo[] newMethodInfos = (RuntimeMethodInfo[])this.tablePin.newArray(RuntimeMethodInfo.class, newTableSize);
        this.tablePin.close();
        if (oldMethodInfos != null) {
            System.arraycopy(oldMethodInfos, 0, newMethodInfos, 0, oldMethodInfos.length);
            oldTablePin.release();
        }
        this.methodInfos = newMethodInfos;
        if (oldMethodInfos != null) {
            Arrays.fill(oldMethodInfos, null);
        }
    }

    protected void invalidateMethod(RuntimeMethodInfo methodInfo) {
        SubstrateInstalledCode installedCode;
        VMOperation.guaranteeInProgress("Modifying code tables that are used by the GC");
        this.invalidateMethodCount.inc();
        assert (this.verifyTable());
        if (Options.TraceCodeCache.getValue().booleanValue()) {
            Log.log().string("[Invalidate method: ");
            RuntimeCodeInfo.logMethod(Log.log(), methodInfo);
            Log.log().string("]").newline();
        }
        if ((installedCode = (SubstrateInstalledCode)methodInfo.installedCode.get()) != null) {
            assert (!installedCode.isValid() || methodInfo.getCodeStart().rawValue() == installedCode.getAddress());
            installedCode.clearAddress();
        }
        InstalledCodeObserverSupport.removeObservers(methodInfo.codeObserverHandles);
        Deoptimizer.deoptimizeInRange(methodInfo.getCodeStart(), methodInfo.getCodeEnd(), false);
        int idx = RuntimeCodeInfo.binarySearch(this.methodInfos, 0, this.numMethods, methodInfo.getCodeStart());
        assert (idx >= 0) : "methodInfo must be in table";
        System.arraycopy(this.methodInfos, idx + 1, this.methodInfos, idx, this.numMethods - (idx + 1));
        --this.numMethods;
        this.methodInfos[this.numMethods] = null;
        Heap.getHeap().getGC().unregisterObjectReferenceWalker(methodInfo.constantsWalker);
        Arrays.fill(methodInfo.frameInfoObjectConstants, null);
        if (methodInfo.frameInfoSourceClasses != null) {
            Arrays.fill(methodInfo.frameInfoSourceClasses, null);
        }
        if (methodInfo.frameInfoSourceMethodNames != null) {
            Arrays.fill(methodInfo.frameInfoSourceMethodNames, null);
        }
        if (methodInfo.frameInfoNames != null) {
            Arrays.fill(methodInfo.frameInfoNames, null);
        }
        methodInfo.allocator.release();
        methodInfo.freeInstalledCode();
        if (Options.TraceCodeCache.getValue().booleanValue()) {
            this.logTable();
        }
        assert (this.verifyTable());
    }

    @Uninterruptible(reason="called from uninterruptible code")
    private boolean verifyTable() {
        int i;
        if (this.methodInfos == null) {
            assert (this.numMethods == 0) : "a1";
            return true;
        }
        assert (this.numMethods <= this.methodInfos.length) : "a11";
        for (i = 0; i < this.numMethods; ++i) {
            RuntimeMethodInfo methodInfo = this.methodInfos[i];
            assert (methodInfo != null) : "a20";
            assert (i == 0 || ((UnsignedWord)this.methodInfos[i - 1].getCodeStart()).belowThan((UnsignedWord)this.methodInfos[i].getCodeStart())) : "a22";
            assert (i == 0 || ((UnsignedWord)this.methodInfos[i - 1].getCodeEnd()).belowOrEqual((UnsignedWord)methodInfo.getCodeStart())) : "a23";
        }
        for (i = this.numMethods; i < this.methodInfos.length; ++i) {
            assert (this.methodInfos[i] == null) : "a31";
        }
        return true;
    }

    public void logTable() {
        this.logTable(Log.log());
    }

    public void logRecentOperations(Log log) {
        log.string("== [Recent RuntimeCodeCache operations: ");
        this.recentCodeCacheOperations.foreach((context, entry) -> Log.log().newline().string((String)entry));
        log.string("]").newline();
    }

    public void logTable(Log log) {
        log.string("== [RuntimeCodeCache: ").signed(this.numMethods).string(" methods");
        for (int i = 0; i < this.numMethods; ++i) {
            log.newline().hex((WordBase)this.methodInfos[i].getCodeStart()).string("  ");
            RuntimeCodeInfo.logMethod(log, this.methodInfos[i]);
        }
        log.string("]").newline();
    }

    private static void logMethod(Log log, RuntimeMethodInfo methodInfo) {
        log.string(methodInfo.name);
        log.string("  ip: ").hex((WordBase)methodInfo.getCodeStart()).string(" - ").hex((WordBase)methodInfo.getCodeEnd());
        log.string("  size: ").unsigned((WordBase)methodInfo.getCodeSize());
    }

    long logMethodOperation(RuntimeMethodInfo methodInfo, String kind) {
        long current = ++this.codeCacheOperationSequenceNumber;
        StringBuilderLog log = new StringBuilderLog();
        log.string(kind).string(": ");
        RuntimeCodeInfo.logMethod(log, methodInfo);
        log.string(" ").unsigned(current).string(":{");
        this.recentCodeCacheOperations.append(log.getResult());
        return current;
    }

    void logMethodOperationEnd(long operationNumber) {
        StringBuilderLog log = new StringBuilderLog();
        log.string("}:").unsigned(operationNumber);
        this.recentCodeCacheOperations.append(log.getResult());
    }

    public boolean walkRuntimeMethods(MemoryWalker.Visitor visitor) {
        VMOperation.guaranteeInProgress("Modifying code tables that are used by the GC");
        boolean continueVisiting = true;
        for (int i = 0; continueVisiting && i < this.numMethods; ++i) {
            continueVisiting = visitor.visitRuntimeCompiledMethod(this.methodInfos[i], (MemoryWalker.RuntimeCompiledMethodAccess)ImageSingletons.lookup(MemoryWalkerAccessImpl.class));
        }
        return continueVisiting;
    }

    public static final class MemoryWalkerAccessImpl
    implements MemoryWalker.RuntimeCompiledMethodAccess<RuntimeMethodInfo> {
        @Platforms(value={Platform.HOSTED_ONLY.class})
        protected MemoryWalkerAccessImpl() {
        }

        @Override
        public UnsignedWord getStart(RuntimeMethodInfo runtimeMethod) {
            return (UnsignedWord)runtimeMethod.getCodeStart();
        }

        @Override
        public UnsignedWord getSize(RuntimeMethodInfo runtimeMethod) {
            return runtimeMethod.getCodeSize();
        }

        @Override
        public String getName(RuntimeMethodInfo runtimeMethod) {
            return runtimeMethod.getName();
        }
    }

    public static class Options {
        @Option(help={"Print logging information for runtime code cache modifications"})
        public static final RuntimeOptionKey<Boolean> TraceCodeCache = new RuntimeOptionKey<Boolean>(false);
    }
}

