WITH FUNCTION shortest_path_function(
  start_nodes_translation      CLOB
, end_nodes_translation        CLOB
, path_connections_translation CLOB
, k                            NUMBER
, min_hops                     NUMBER   default 0
, max_hops                     NUMBER   default 9223372036854775807
)
RETURN xmltype IS
  start_nodes_view      VARCHAR2(4000) := 'snv_' || rawtohex(sys_guid());
  end_nodes_view        VARCHAR2(4000) := 'env_' || rawtohex(sys_guid());
  path_connections_view VARCHAR2(4000) := 'pcv_' || rawtohex(sys_guid());
  shortest_paths_table  VARCHAR2(4000) := 'spt_' || rawtohex(sys_guid());
  start_curs            SYS_REFCURSOR;
  start_src_table       VARCHAR2(4000);
  start_src_key         VARCHAR2(4000);
  end_dst_table         VARCHAR2(4000);
  end_dst_key           VARCHAR2(4000);
  end_curs              SYS_REFCURSOR;
  level_start           NUMBER;
  level_end             NUMBER;
  num_hops              NUMBER;
  top_k_reached         BOOLEAN;
  vertex_cnt            NUMBER;
  idx                   NUMBER;
  current_vertex_table  VARCHAR2(4000);
  current_vertex_key    VARCHAR2(4000);
  num_paths             NUMBER;
  neighbors_curs        SYS_REFCURSOR;
  dst_table             VARCHAR2(4000);
  dst_key               VARCHAR2(4000);
  exp                   VARCHAR2(4000);
  previous_idx          NUMBER;
  return_curs           SYS_REFCURSOR;
  exp_path              VARCHAR2(4000);
  paths_found           BOOLEAN;
  return_xml            XMLTYPE;

  TYPE num_list   IS TABLE OF NUMBER INDEX BY VARCHAR2(4000);
  num_paths_list        NUM_LIST;

  TYPE num_array  IS VARRAY(2147483647) OF NUMBER;
  TYPE str_array  IS VARRAY(2147483647) OF VARCHAR2(4000);
  dst_table_queue       STR_ARRAY := STR_ARRAY();
  dst_key_queue         STR_ARRAY := STR_ARRAY();
  exp_queue             STR_ARRAY := STR_ARRAY();
  prev_idx_queue        NUM_ARRAY := NUM_ARRAY();
  num_hops_queue        NUM_ARRAY := NUM_ARRAY();

  pragma autonomous_transaction;
BEGIN
  -- Create views
  execute immediate 
    'CREATE VIEW ' || start_nodes_view || ' AS ' || start_nodes_translation;

  execute immediate 
    'CREATE VIEW ' || end_nodes_view || ' AS ' || end_nodes_translation;

  execute immediate 
    'CREATE VIEW ' || path_connections_view || ' AS ' || path_connections_translation;

  -- Create table with shortest paths
  execute immediate
    'CREATE TABLE ' || shortest_paths_table || '(
       src_table    varchar2(4000)
     , src_key      varchar2(4000)
     , dst_table    varchar2(4000)
     , dst_key      varchar2(4000)
     , exp_path     varchar2(4000)
     )';

  paths_found := false;

  -- Find paths for every start node
  open start_curs for
    'SELECT "src_table", "src_key" FROM ' || start_nodes_view;
  fetch start_curs into start_src_table, start_src_key;

  while start_curs%FOUND loop
    -- Initialize data structures
    vertex_cnt := 1;
    num_paths_list.delete;
    dst_table_queue.delete;
    dst_key_queue.delete;
    exp_queue.delete;
    prev_idx_queue.delete;
    num_hops_queue.delete;

    -- Insert 0-length path to start node
    dst_table_queue.extend();
    dst_table_queue(vertex_cnt) := start_src_table;
    dst_key_queue.extend();
    dst_key_queue(vertex_cnt) := start_src_key;
    exp_queue.extend();
    exp_queue(vertex_cnt) := '';
    prev_idx_queue.extend();
    prev_idx_queue(vertex_cnt) := -1;
    num_hops_queue.extend();
    num_hops_queue(vertex_cnt) := 0;
    vertex_cnt := vertex_cnt + 1;
    if min_hops = 0 then
      num_paths_list(start_src_table || '|' || start_src_key) := 1;
    end if;

    -- Expand reachability graph from start node until we reach
    -- all end nodes k times or until we cannot expand anymore
    level_start := 1;
    level_end := 1;

    open end_curs for
      'SELECT "dst_table", "dst_key" FROM ' || end_nodes_view;
    fetch end_curs into end_dst_table, end_dst_key;
    while end_curs%FOUND AND level_start <= level_end loop

      -- Find if we already have k paths to end node
      top_k_reached := false;
      if level_start <= level_end then
        BEGIN
          top_k_reached := num_paths_list(end_dst_table || '|' || end_dst_key) = k;
         EXCEPTION when NO_DATA_FOUND then null;
        END;
      end if;

      -- Find paths to end node
      while (level_start <= level_end AND NOT top_k_reached) loop

        -- Loop through each vertex in current level
        idx := level_start;
        while (idx <= level_end AND NOT top_k_reached) loop

          current_vertex_table := dst_table_queue(idx);
          current_vertex_key := dst_key_queue(idx);
          num_hops := num_hops_queue(idx);

          if num_hops < max_hops then
            -- Get neighbors of current vertex
            open neighbors_curs for
              'SELECT pc."dst_table", pc."dst_key", pc."exp"
               FROM ' || path_connections_view || ' pc
               WHERE pc."src_table" = :current_vertex_table
                 AND pc."src_key" = :current_vertex_key'
            using current_vertex_table, current_vertex_key;

            -- Loop through each neighbor
            fetch neighbors_curs into dst_table, dst_key, exp;
            while neighbors_curs%FOUND loop

              -- Find if there are k paths to the neighbor
              BEGIN
                num_paths := num_paths_list(dst_table || '|' || dst_key);
               EXCEPTION WHEN NO_DATA_FOUND THEN num_paths := 0;
              END;

              if num_paths < k then
                -- Add neighbor to the queue
                dst_table_queue.extend();
                dst_table_queue(vertex_cnt) := dst_table;
                dst_key_queue.extend();
                dst_key_queue(vertex_cnt) := dst_key;
                exp_queue.extend();
                exp_queue(vertex_cnt) := exp;
                prev_idx_queue.extend();
                prev_idx_queue(vertex_cnt) := idx;
                num_hops_queue.extend();
                num_hops_queue(vertex_cnt) := num_hops + 1;
                vertex_cnt := vertex_cnt + 1;

                -- Update number of paths found to the neighbor
                if min_hops <= num_hops_queue(vertex_cnt - 1) then
                  num_paths := num_paths + 1;
                  num_paths_list(dst_table || '|' || dst_key) := num_paths;
                end if;
              end if;

              -- Find if neighbor is current destination and we have reached k
              if (dst_table = end_dst_table AND dst_key = end_dst_key AND num_paths = k) then
                top_k_reached := true;
              end if;

              fetch neighbors_curs into dst_table, dst_key, exp;
            end loop;
            close neighbors_curs;

          end if;
          idx:= idx + 1;
        end loop;

        -- If we found k paths for current destination, we keep expanding
        -- same level for the next destination node
        if top_k_reached then
          level_start := idx;
        else
          level_start := level_end + 1;
        end if;

        -- If we are already at the next level, update end index of the level
        if level_start = level_end + 1 then
          level_end := vertex_cnt - 1;
        end if;

      end loop;
      fetch end_curs into end_dst_table, end_dst_key;
    end loop;
    close end_curs;

    -- Loop through the queue to build and store found shortest paths
    for i in 1..(vertex_cnt - 1) loop
      end_dst_table := dst_table_queue(i);
      end_dst_key := dst_key_queue(i);
      exp := exp_queue(i);
      previous_idx := prev_idx_queue(i);
      num_hops := num_hops_queue(i);

      if num_hops >= min_hops then
        BEGIN
          -- Find if element at the queue is a destination node
          execute immediate
            'select "dst_table", "dst_key" from ' || end_nodes_view || '
             where "dst_table" = :end_dst_table
               and "dst_key" = :end_dst_key'
          using end_dst_table, end_dst_key;

         -- Build path
          exp_path := exp || '</EXP_PATH>';
          while previous_idx <> -1 loop
            exp := exp_queue(previous_idx);
            exp_path := exp || exp_path;
            previous_idx := prev_idx_queue(previous_idx);
          end loop;
          exp_path := '<EXP_PATH>' || exp_path;

          -- Insert path
          execute immediate
            'INSERT INTO ' || shortest_paths_table || ' (src_table, src_key, dst_table, dst_key, exp_path)
             VALUES (:src_table, :src_key, :dst_table, :dst_key, :exp_path)'
          using start_src_table, start_src_key, end_dst_table, end_dst_key, exp_path;
          paths_found := true;
         EXCEPTION when NO_DATA_FOUND then null;
        END;
      end if;

    end loop;

    fetch start_curs into start_src_table, start_src_key;
  end loop;
  close start_curs;

  -- Drop views
  BEGIN
    execute immediate 'drop view ' || start_nodes_view;
   EXCEPTION when others then null;
  END;

  BEGIN
    execute immediate 'drop view ' || end_nodes_view;
   EXCEPTION when others then null;
  END;

  BEGIN
    execute immediate 'drop view ' || path_connections_view;
   EXCEPTION when others then null;
  END;

  -- Create return xmltype
  if paths_found then
    OPEN return_curs for
      'SELECT src_table, src_key,
              dst_table, dst_key,
              exp_path
       FROM ' || shortest_paths_table;

    return_xml := xmltype(return_curs);
    CLOSE return_curs;
  else
    return_xml := xmltype('<?xml version="1.0"?><ROWSET></ROWSET>');
  end if;

  -- Drop table with shortest paths
  BEGIN
    execute immediate 'drop table ' || shortest_paths_table;
   EXCEPTION when others then null;
  END;  

  return return_xml;

EXCEPTION when others then
  
  -- Drop views
  BEGIN
    execute immediate 'drop view ' || start_nodes_view;
   EXCEPTION when others then null;
  END;

  BEGIN
    execute immediate 'drop view ' || end_nodes_view;
   EXCEPTION when others then null;
  END;

  BEGIN
    execute immediate 'drop view ' || path_connections_view;
   EXCEPTION when others then null;
  END;

  -- Drop table with shortest paths
  BEGIN
    execute immediate 'drop table ' || shortest_paths_table;
   EXCEPTION when others then null;
  END;  

  -- Raise exception
  raise;
END shortest_path_function;
