/* 
 * This file is part of the Echo Web Application Framework (hereinafter "Echo").
 * Copyright (C) 2002-2009 NextApp, Inc.
 *
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 */

package nextapp.echo.app.button;

import java.util.EventListener;

import nextapp.echo.app.Alignment;
import nextapp.echo.app.Extent;
import nextapp.echo.app.ImageReference;
import nextapp.echo.app.event.ChangeEvent;
import nextapp.echo.app.event.ChangeListener;

/**
 * An abstract base class for on/off toggle button components.
 */
public abstract class ToggleButton extends AbstractButton {
    
    /** Serial Version UID. */
    private static final long serialVersionUID = 20070101L;

    public static final String PROPERTY_DISABLED_STATE_ICON = "disabledStateIcon";
    public static final String PROPERTY_DISABLED_SELECTED_STATE_ICON = "disabledSelectedStateIcon";
    public static final String PROPERTY_PRESSED_STATE_ICON = "pressedStateIcon";
    public static final String PROPERTY_PRESSED_SELECTED_STATE_ICON = "pressedSelectedStateIcon";
    public static final String PROPERTY_ROLLOVER_STATE_ICON = "rolloverStateIcon";
    public static final String PROPERTY_ROLLOVER_SELECTED_STATE_ICON = "rolloverSelectedStateIcon";
    public static final String PROPERTY_STATE_ALIGNMENT = "stateAlignment";
    public static final String PROPERTY_STATE_ICON = "stateIcon";
    public static final String PROPERTY_STATE_MARGIN = "stateMargin";
    public static final String PROPERTY_STATE_POSITION = "statePosition";
    public static final String PROPERTY_SELECTED_STATE_ICON = "selectedStateIcon";
    
    public static final String SELECTED_CHANGED_PROPERTY = "selected";
    
    /**
     * Forwards events generated by the model to listeners registered with the
     * component instance.
     */
    private ChangeListener changeForwarder = new ChangeListener() {

        /** Serial Version UID. */
        private static final long serialVersionUID = 20070101L;
    
        /**
         * @see nextapp.echo.app.event.ChangeListener#stateChanged(nextapp.echo.app.event.ChangeEvent)
         */
        public void stateChanged(ChangeEvent e) {
            fireStateChanged();
            if (isSelected()) {
                firePropertyChange(SELECTED_CHANGED_PROPERTY, Boolean.FALSE, Boolean.TRUE);
            } else {
                firePropertyChange(SELECTED_CHANGED_PROPERTY, Boolean.TRUE, Boolean.FALSE);
            }
        }
    };

    /**
     * Adds a <code>ChangeListener</code> to receive notification of state
     * changes, i.e., to the selected state of a <code>ToggleButton</code>.
     * 
     * @param l  the listener to add
     */
    public void addChangeListener(ChangeListener l) {
        getEventListenerList().addListener(ChangeListener.class, l);
    }
    
    /**
     * Notifies all listeners that have registered for this event type.
     */
    public void fireStateChanged() {
        EventListener[] listeners = getEventListenerList().getListeners(ChangeListener.class);
        ChangeEvent e = new ChangeEvent(this);
        for (int index = 0; index < listeners.length; ++index) {
            ((ChangeListener) listeners[index]).stateChanged(e);
        }
    }
    
    /**
     * Returns the selected state icon when the button is disabled.
     * 
     * @return the icon
     */
    public ImageReference getDisabledSelectedStateIcon() {
        return (ImageReference) get(PROPERTY_DISABLED_SELECTED_STATE_ICON);
    }
    
    /**
     * Returns the default (non-selected) state icon when the button is disabled.
     * 
     * @return the icon
     */
    public ImageReference getDisabledStateIcon() {
        return (ImageReference) get(PROPERTY_DISABLED_STATE_ICON);
    }
    
    /**
     * Returns the selected state icon displayed when the 
     * button is being pressed.
     * 
     * @return the icon
     */
    public ImageReference getPressedSelectedStateIcon() {
        return (ImageReference) get(PROPERTY_PRESSED_SELECTED_STATE_ICON);
    }
    
    /**
     * Returns the default (non-selected) state icon displayed when the 
     * button is being pressed.
     * 
     * @return the icon
     */
    public ImageReference getPressedStateIcon() {
        return (ImageReference) get(PROPERTY_PRESSED_STATE_ICON);
    }
    
    /**
     * Returns the selected state icon displayed when the mouse
     * cursor is inside the bounds of the button.
     * 
     * @return the icon
     */
    public ImageReference getRolloverSelectedStateIcon() {
        return (ImageReference) get(PROPERTY_ROLLOVER_SELECTED_STATE_ICON);
    }
    
    /**
     * Returns the default (non-selected) state icon displayed when the mouse
     * cursor is inside the bounds of the button.
     * 
     * @return the icon
     */
    public ImageReference getRolloverStateIcon() {
        return (ImageReference) get(PROPERTY_ROLLOVER_STATE_ICON);
    }
    
    /**
     * Returns the selected state icon.
     * 
     * @return the icon
     */
    public ImageReference getSelectedStateIcon() {
        return (ImageReference) get(PROPERTY_SELECTED_STATE_ICON);
    }
    
    /**
     * Returns the alignment of the state icon relative to the button's
     * text/icon.
     * 
     * @return the state alignment
     */
    public Alignment getStateAlignment() {
        return (Alignment) get(PROPERTY_STATE_ALIGNMENT);
    }

    /**
     * Returns the default (non-selected) state icon.
     * 
     * @return the icon
     */
    public ImageReference getStateIcon() {
        return (ImageReference) get(PROPERTY_STATE_ICON);
    }
    
    /**
     * Returns the margin size between the state icon and the 
     * toggle button's icon and/or text.  The margin will only
     * be displayed in the case where icon and/or text are
     * present.
     * This property only supports <code>Extent</code>s with
     * fixed (i.e., not percent) units.
     * 
     * @return the margin size 
     */
    public Extent getStateMargin() {
        return (Extent) get(PROPERTY_STATE_MARGIN);
    }

    /**
     * Returns the position of the state icon relative to the button's
     * text/icon.
     * 
     * @return the state position
     */
    public Alignment getStatePosition() {
        return (Alignment) get(PROPERTY_STATE_POSITION);
    }
    
    /**
     * Determines the selection state.
     * 
     * @return the selection state
     */
    public boolean isSelected() {
        return ((ToggleButtonModel) getModel()).isSelected();
    }

    /**
     * @see nextapp.echo.app.Component#processInput(java.lang.String, java.lang.Object)
     */
    public void processInput(String name, Object value) {
        super.processInput(name, value);
        if (SELECTED_CHANGED_PROPERTY.equals(name)) {
            setSelected(((Boolean) value).booleanValue());
        }
    }

    /**
     * Removes a <code>ChangeListener</code> from being notified of state
     * changes, i.e., to the selected state of a <code>ToggleButton</code>.
     * 
     * @param l the listener to remove
     */
    public void removeChangeListener(ChangeListener l) {
        getEventListenerList().removeListener(ChangeListener.class, l);
    }
    
    /**
     * Sets the selected state icon when the button is disabled.
     * 
     * @param newValue the new icon
     */
    public void setDisabledSelectedStateIcon(ImageReference newValue) {
        set(PROPERTY_DISABLED_SELECTED_STATE_ICON, newValue);
    }
    
    /**
     * Sets the default (non-selected) state icon when the button is disabled.
     * 
     * @param newValue the new icon
     */
    public void setDisabledStateIcon(ImageReference newValue) {
        set(PROPERTY_DISABLED_STATE_ICON, newValue);
    }
    
    /**
     * @see nextapp.echo.app.button.AbstractButton#setModel(nextapp.echo.app.button.ButtonModel)
     */
    public void setModel(ButtonModel newValue) {
        if (!(newValue instanceof ToggleButtonModel)) {
            throw new IllegalArgumentException("Model is not a ToggleButtonModel.");
        }

        ToggleButtonModel oldValue = (ToggleButtonModel) getModel();

        if (oldValue != null) {
            oldValue.removeChangeListener(changeForwarder);
        }
        
        super.setModel(newValue);

        ((ToggleButtonModel) newValue).addChangeListener(changeForwarder);
    }

    /**
     * Sets the selected state icon displayed when the 
     * button is being pressed.
     * 
     * @param newValue the new icon
     */
    public void setPressedSelectedStateIcon(ImageReference newValue) {
        set(PROPERTY_PRESSED_SELECTED_STATE_ICON, newValue);
    }
    
    /**
     * Sets the default (non-selected) state icon displayed when the 
     * button is being pressed.
     * 
     * @param newValue the new icon
     */
    public void setPressedStateIcon(ImageReference newValue) {
        set(PROPERTY_PRESSED_STATE_ICON, newValue);
    }
    
    /**
     * Sets the selected state icon displayed when the mouse
     * cursor is inside the bounds of the button.
     * 
     * @param newValue the new icon
     */
    public void setRolloverSelectedStateIcon(ImageReference newValue) {
        set(PROPERTY_ROLLOVER_SELECTED_STATE_ICON, newValue);
    }
    
    /**
     * Sets the default (non-selected) state icon displayed when the mouse
     * cursor is inside the bounds of the button.
     * 
     * @param newValue the new icon
     */
    public void setRolloverStateIcon(ImageReference newValue) {
        set(PROPERTY_ROLLOVER_STATE_ICON, newValue);
    }
    
    /**
     * Sets the selected state icon.
     * 
     * @param newValue the new icon
     */
    public void setSelectedStateIcon(ImageReference newValue) {
        set(PROPERTY_SELECTED_STATE_ICON, newValue);
    }
    
    /**
     * Sets the selection state.
     * 
     * @param newValue the new selection state
     */
    public void setSelected(boolean newValue) {
        ((ToggleButtonModel) getModel()).setSelected(newValue);
    }
    
    /**
     * Sets the alignment of the state icon relative to the text/icon
     * of the button.
     * Note that only one of the provided <code>Alignment</code>'s
     * settings should be non-default.
     * 
     * @param newValue the new state alignment
     */
    public void setStateAlignment(Alignment newValue) {
        set(PROPERTY_STATE_ALIGNMENT, newValue);
    }
    
    /**
     * Sets the default (non-selected) state icon.
     * 
     * @param newValue the new icon
     */
    public void setStateIcon(ImageReference newValue) {
        set(PROPERTY_STATE_ICON, newValue);
    }
    
    /**
     * Sets the margin size between the state icon and the 
     * toggle button's icon and/or text.  The margin will only
     * be displayed in the case where icon and/or text are
     * present.
     * This property only supports <code>Extent</code>s with
     * fixed (i.e., not percent) units.
     * 
     * @param newValue the new margin size 
     */
    public void setStateMargin(Extent newValue) {
        set(PROPERTY_STATE_MARGIN, newValue);
    }

    /**
     * Sets the position of the state icon relative to the text/icon
     * of the button.
     * Note that only one of the provided <code>Alignment</code>'s
     * settings should be non-default.
     * 
     * @param newValue the new state position
     */
    public void setStatePosition(Alignment newValue) {
        set(PROPERTY_STATE_POSITION, newValue);
    }
}
