package com.netease.nimlib.log.core;

import android.text.TextUtils;
import android.util.Log;

import com.netease.nimlib.log.sdk.LogBase;
import com.netease.nimlib.log.sdk.util.FileUtils;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.UnsupportedEncodingException;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;

/**
 * Created by huangjun on 2017/5/25.
 * 头四个字节记录当前写入的有效数据偏移
 */

public class MMapWriter {
    private static final String TAG = "MMapWriter";
    private static final boolean OUTPUT_LOG = false;
    private static final int K = 1024;
    private static final int M = 1024 * K;

    private static final int HEADER_LENGTH = 4;
    private static final int COPY_BUFFER_SIZE = M;
    private static final int MAX_PRE_WRITE_LINES = 100;

    private static final int DEFAULT_MAX_LENGTH = 8 * M;
    private static final int DEFAULT_BASE_LENGTH = 4 * M;
    private static final int DEFAULT_DANGEROUS_BUFFER_LENGTH = 2 * K;

    private final int MAX_BUFFER_SIZE; // 8M
    private final int BASE_BUFFER_SIZE; // 4M
    private final int LAST_DANGEROUS_BUFFER_SIZE; // 2K
    private RandomAccessFile out;
    private MappedByteBuffer mappedByteBuffer;
    private File destFile;
    private int linesCount = 0;

    public MMapWriter(final int maxLength, final int baseLength, final int lastDangerousLength) {
        MAX_BUFFER_SIZE = (maxLength > 0 && maxLength > baseLength) ? maxLength : DEFAULT_MAX_LENGTH;
        BASE_BUFFER_SIZE = (baseLength > 0 && baseLength < maxLength) ? baseLength : DEFAULT_BASE_LENGTH;
        LAST_DANGEROUS_BUFFER_SIZE = (lastDangerousLength > 0 && lastDangerousLength < baseLength) ?
                lastDangerousLength : DEFAULT_DANGEROUS_BUFFER_LENGTH;
    }

    public boolean open(final String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return false;
        }

        if (checkValid()) {
            close();
        }

        try {
            // dest file
            destFile = FileUtils.getFile(filePath);
            if (destFile == null) {
                log("file path is invalid, path=" + filePath);
                return false;
            }

            // open
            log("try to open file, path=" + destFile.getCanonicalPath());
            out = new RandomAccessFile(destFile, "rw");
            if (out.length() <= 0) {
                out.setLength(MAX_BUFFER_SIZE);
            }

            // map
            mappedByteBuffer = out.getChannel().map(FileChannel.MapMode.READ_WRITE, 0, MAX_BUFFER_SIZE);

            // offset
            int offset = moveToWriteOffset();

            // wipe the remaining data which may be the dirty content
            writeEOF(offset, MAX_BUFFER_SIZE - 1);
            mappedByteBuffer.position(offset);

            // shrink
            offset = shrink(offset);

            // log
            log("open file success, path=" + destFile.getCanonicalPath() + ", offset=" + offset + ", length=" + destFile.length());
        } catch (IOException e) {
            e.printStackTrace();
            log("open file error, e=" + e.getMessage());
        }

        return true;
    }

    public void write(String content) {
        if (TextUtils.isEmpty(content)) {
            return;
        }

        if (destFile == null) {
            log("dest file is null, write failed");
            return;
        }

        // check status
        if (!checkValid()) {
            log("MMapWriter is invalid when do write");
            return;
        }

        byte[] bytes = null;
        try {
            bytes = content.getBytes("UTF8");
            for (int i = 0; i < bytes.length; i++) {
                if (bytes[i] == 0x00) {
                    bytes[i] = 0x20;
                }
            }
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }

        if (bytes == null) {
            return;
        }

        // check position
        int contentLength = bytes.length;
        int offset = mappedByteBuffer.position();
        if (offset >= MAX_BUFFER_SIZE - LAST_DANGEROUS_BUFFER_SIZE || offset + contentLength >= MAX_BUFFER_SIZE - 1) {
            log("mapped buffer has nearly used up, position=" + mappedByteBuffer.position()
                    + "/" + MAX_BUFFER_SIZE + ", will add " + contentLength + ", begin to shrink...");
            shrink(offset);
        }

        // write
        mappedByteBuffer.put(bytes);

        // save offset
        saveWriteOffset();

        // flush
        checkFlush();

        // log
        log("write position " + mappedByteBuffer.position() + "/" + MAX_BUFFER_SIZE + ", add " + contentLength);

        content = content.substring(0, content.length() - 2);
        log("write content : " + content);
    }

    public void close() {
        // force write
        if (mappedByteBuffer != null) {
            mappedByteBuffer.force();
            mappedByteBuffer.clear();
            mappedByteBuffer = null;
        }

        // file channel close
        FileUtils.close(out);

        // log
        log("file close success");
    }

    /**
     * 文件裁剪
     */
    private int shrink(int offset) {
        // check status
        if (!checkValid()) {
            log("MMapWriter is invalid when do shrink");
            return offset;
        }

        if (offset < MAX_BUFFER_SIZE - LAST_DANGEROUS_BUFFER_SIZE) {
            return offset; // remain buffer is enough
        }

        // find first line start offset, max find 1K bytes
        int tempReadStart = MAX_BUFFER_SIZE - LAST_DANGEROUS_BUFFER_SIZE - BASE_BUFFER_SIZE;
        int readCount = 0;
        mappedByteBuffer.position(tempReadStart);
        while (true) {
            if (mappedByteBuffer.get() == '\r' && mappedByteBuffer.get() == '\n' || readCount > K) {
                break;
            }
            readCount++;
        }

        final int readStart = mappedByteBuffer.position();
        final int readEnd = offset - 1;
        int readPos = readStart;
        int writePos = HEADER_LENGTH;
        int readLength;
        int remainLength;

        mappedByteBuffer.position(readPos);
        byte buf[] = null;
        while (readPos <= readEnd) {
            // read
            remainLength = readEnd - readPos + 1;
            readLength = remainLength < COPY_BUFFER_SIZE ? remainLength : COPY_BUFFER_SIZE;
            if (buf == null || buf.length != readLength) {
                buf = new byte[readLength];
            }
            mappedByteBuffer.get(buf);
            readPos = mappedByteBuffer.position();

            // write
            mappedByteBuffer.position(writePos);
            mappedByteBuffer.put(buf);
            writePos = mappedByteBuffer.position();

            // save write offset
            saveWriteOffset();

            // prepare next read
            mappedByteBuffer.position(readPos);
        }

        // wipe remain space
        writeEOF(writePos, readEnd);

        // force
        mappedByteBuffer.force();

        // offset
        mappedByteBuffer.position(writePos);

        // log
        log("shrink file success, new offset=" + mappedByteBuffer.position());

        return mappedByteBuffer.position();
    }

    private void writeEOF(int startOffset, int end) {
        if (startOffset < HEADER_LENGTH) {
            return;
        }

        mappedByteBuffer.position(startOffset);

        final byte b = 0x00;
        byte buf[] = null;
        int writeLength;
        int remainLength;
        while (mappedByteBuffer.position() <= end) {
            remainLength = end - mappedByteBuffer.position() + 1;
            writeLength = remainLength < COPY_BUFFER_SIZE ? remainLength : COPY_BUFFER_SIZE;

            // zero memory
            if (buf == null || buf.length != writeLength) {
                buf = new byte[writeLength];
                Arrays.fill(buf, b);
            }

            // write
            mappedByteBuffer.put(buf);
        }

        log("write EOF from " + startOffset + " to " + end);
    }

    private int moveToWriteOffset() {
        mappedByteBuffer.position(0);
        int offset = mappedByteBuffer.getInt();
        if (offset >= HEADER_LENGTH && offset < MAX_BUFFER_SIZE) {
            mappedByteBuffer.position(offset);
            return offset;
        } else {
            mappedByteBuffer.position(0);
            return saveWriteOffset();
        }
    }

    private int saveWriteOffset() {
        int offset = mappedByteBuffer.position();
        if (offset < HEADER_LENGTH) {
            offset = HEADER_LENGTH;
        }

        mappedByteBuffer.position(0);
        mappedByteBuffer.putInt(offset);
        mappedByteBuffer.position(offset);

        return offset;
    }

    public void forceFlush() {
        if (!checkValid()) {
            return;
        }

        mappedByteBuffer.force();
        linesCount = 0;
        log("MappedByteBuffer force flush ");
    }

    private void checkFlush() {
        if (++linesCount >= MAX_PRE_WRITE_LINES && checkValid()) {
            mappedByteBuffer.force();
            linesCount = 0;
            log("MappedByteBuffer flush ");
        }
    }

    private boolean checkValid() {
        return out != null && mappedByteBuffer != null;
    }

    /**
     * ******************************* log ********************************
     */

    /**
     * 设置调试xlog需要的日志
     *
     * @param log 调试用的日志
     */
    public void setDebugLog(LogBase log) {
        logBase = log;
    }

    private LogBase logBase;

    private void log(String content) {
        if (OUTPUT_LOG) {
            if (logBase != null) {
                logBase.i(TAG, content);
            } else {
                Log.i(TAG, content);
            }
        }
    }
}
