"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PlaywrightClient = void 0;

var _ws = _interopRequireDefault(require("ws"));

var _connection = require("../client/connection");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
class PlaywrightClient {
  static async connect(options) {
    const {
      wsEndpoint,
      forwardPorts,
      timeout = 30000
    } = options;
    const connection = new _connection.Connection();
    const ws = new _ws.default(wsEndpoint);

    connection.onmessage = message => ws.send(JSON.stringify(message));

    ws.on('message', message => connection.dispatch(JSON.parse(message.toString())));
    const errorPromise = new Promise((_, reject) => ws.on('error', error => reject(error)));
    const closePromise = new Promise((_, reject) => ws.on('close', () => reject(new Error('Connection closed'))));
    const playwrightClientPromise = new Promise(async (resolve, reject) => {
      const playwright = await connection.waitForObjectWithKnownName('Playwright');
      if (forwardPorts) await playwright._enablePortForwarding(forwardPorts).catch(reject);
      resolve(new PlaywrightClient(playwright, ws));
    });
    let timer;

    try {
      await Promise.race([playwrightClientPromise, errorPromise, closePromise, new Promise((_, reject) => timer = setTimeout(() => reject(`Timeout of ${timeout}ms exceeded while connecting.`), timeout))]);
      return await playwrightClientPromise;
    } finally {
      clearTimeout(timer);
    }
  }

  constructor(playwright, ws) {
    this._playwright = void 0;
    this._ws = void 0;
    this._closePromise = void 0;
    this._playwright = playwright;
    this._ws = ws;
    this._closePromise = new Promise(f => ws.on('close', f));
  }

  playwright() {
    return this._playwright;
  }

  async close() {
    this._ws.close();

    await this._closePromise;
  }

}

exports.PlaywrightClient = PlaywrightClient;