"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RecorderSupplement = void 0;

var fs = _interopRequireWildcard(require("fs"));

var _codeGenerator = require("./recorder/codeGenerator");

var _utils = require("./recorder/utils");

var _page = require("../page");

var _frames = require("../frames");

var _browserContext = require("../browserContext");

var _java = require("./recorder/java");

var _javascript = require("./recorder/javascript");

var _csharp = require("./recorder/csharp");

var _python = require("./recorder/python");

var recorderSource = _interopRequireWildcard(require("../../generated/recorderSource"));

var consoleApiSource = _interopRequireWildcard(require("../../generated/consoleApiSource"));

var _recorderApp = require("./recorder/recorderApp");

var _instrumentation = require("../instrumentation");

var _utils2 = require("../../utils/utils");

var _recorderUtils = require("./recorder/recorderUtils");

var _debugger = require("./debugger");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const symbol = Symbol('RecorderSupplement');

class RecorderSupplement {
  static showInspector(context) {
    RecorderSupplement.show(context, {}).catch(() => {});
  }

  static show(context, params = {}) {
    let recorderPromise = context[symbol];

    if (!recorderPromise) {
      const recorder = new RecorderSupplement(context, params);
      recorderPromise = recorder.install().then(() => recorder);
      context[symbol] = recorderPromise;
    }

    return recorderPromise;
  }

  constructor(context, params) {
    this._generator = void 0;
    this._pageAliases = new Map();
    this._lastPopupOrdinal = 0;
    this._lastDialogOrdinal = 0;
    this._lastDownloadOrdinal = 0;
    this._timers = new Set();
    this._context = void 0;
    this._mode = void 0;
    this._highlightedSelector = '';
    this._recorderApp = null;
    this._params = void 0;
    this._currentCallsMetadata = new Map();
    this._recorderSources = void 0;
    this._userSources = new Map();
    this._allMetadatas = new Map();
    this._debugger = void 0;
    this._context = context;
    this._debugger = _debugger.Debugger.lookup(context);
    context.instrumentation.addListener(this);
    this._params = params;
    this._mode = params.startRecording ? 'recording' : 'none';
    const language = params.language || context._options.sdkLanguage;
    const languages = new Set([new _java.JavaLanguageGenerator(), new _javascript.JavaScriptLanguageGenerator(false), new _javascript.JavaScriptLanguageGenerator(true), new _python.PythonLanguageGenerator(false), new _python.PythonLanguageGenerator(true), new _csharp.CSharpLanguageGenerator()]);
    const primaryLanguage = [...languages].find(l => l.id === language);
    if (!primaryLanguage) throw new Error(`\n===============================\nUnsupported language: '${language}'\n===============================\n`);
    languages.delete(primaryLanguage);
    const orderedLanguages = [primaryLanguage, ...languages];
    this._recorderSources = [];
    const generator = new _codeGenerator.CodeGenerator(context._browser.options.name, !!params.startRecording, params.launchOptions || {}, params.contextOptions || {}, params.device, params.saveStorage);
    let text = '';
    generator.on('change', () => {
      var _this$_recorderApp;

      this._recorderSources = [];

      for (const languageGenerator of orderedLanguages) {
        const source = {
          file: languageGenerator.fileName,
          text: generator.generateText(languageGenerator),
          language: languageGenerator.highlighter,
          highlight: []
        };
        source.revealLine = source.text.split('\n').length - 1;

        this._recorderSources.push(source);

        if (languageGenerator === orderedLanguages[0]) text = source.text;
      }

      this._pushAllSources();

      (_this$_recorderApp = this._recorderApp) === null || _this$_recorderApp === void 0 ? void 0 : _this$_recorderApp.setFile(primaryLanguage.fileName);
    });

    if (params.outputFile) {
      context.on(_browserContext.BrowserContext.Events.BeforeClose, () => {
        fs.writeFileSync(params.outputFile, text);
        text = '';
      });
      process.on('exit', () => {
        if (text) fs.writeFileSync(params.outputFile, text);
      });
    }

    this._generator = generator;
  }

  async install() {
    const recorderApp = await _recorderApp.RecorderApp.open(this._context);
    this._recorderApp = recorderApp;
    recorderApp.once('close', () => {
      this._debugger.resume(false);

      this._recorderApp = null;
    });
    recorderApp.on('event', data => {
      if (data.event === 'setMode') {
        this._setMode(data.params.mode);

        this._refreshOverlay();

        return;
      }

      if (data.event === 'selectorUpdated') {
        this._highlightedSelector = data.params.selector;

        this._refreshOverlay();

        return;
      }

      if (data.event === 'step') {
        this._debugger.resume(true);

        return;
      }

      if (data.event === 'resume') {
        this._debugger.resume(false);

        return;
      }

      if (data.event === 'pause') {
        this._debugger.pauseOnNextStatement();

        return;
      }

      if (data.event === 'clear') {
        this._clearScript();

        return;
      }
    });
    await Promise.all([recorderApp.setMode(this._mode), recorderApp.setPaused(this._debugger.isPaused()), this._pushAllSources()]);

    this._context.on(_browserContext.BrowserContext.Events.Page, page => this._onPage(page));

    for (const page of this._context.pages()) this._onPage(page);

    this._context.once(_browserContext.BrowserContext.Events.Close, () => {
      for (const timer of this._timers) clearTimeout(timer);

      this._timers.clear();

      recorderApp.close().catch(() => {});
    }); // Input actions that potentially lead to navigation are intercepted on the page and are
    // performed by the Playwright.


    await this._context.exposeBinding('_playwrightRecorderPerformAction', false, (source, action) => this._performAction(source.frame, action), 'utility'); // Other non-essential actions are simply being recorded.

    await this._context.exposeBinding('_playwrightRecorderRecordAction', false, (source, action) => this._recordAction(source.frame, action), 'utility');
    await this._context.exposeBinding('_playwrightRecorderState', false, source => {
      let actionSelector = this._highlightedSelector;
      let actionPoint;

      for (const [metadata, sdkObject] of this._currentCallsMetadata) {
        if (source.page === sdkObject.attribution.page) {
          actionPoint = metadata.point || actionPoint;
          actionSelector = actionSelector || metadata.params.selector;
        }
      }

      const uiState = {
        mode: this._mode,
        actionPoint,
        actionSelector
      };
      return uiState;
    }, 'utility');
    await this._context.exposeBinding('_playwrightRecorderSetSelector', false, async (_, selector) => {
      var _this$_recorderApp2, _this$_recorderApp3;

      this._setMode('none');

      await ((_this$_recorderApp2 = this._recorderApp) === null || _this$_recorderApp2 === void 0 ? void 0 : _this$_recorderApp2.setSelector(selector, true));
      await ((_this$_recorderApp3 = this._recorderApp) === null || _this$_recorderApp3 === void 0 ? void 0 : _this$_recorderApp3.bringToFront());
    }, 'utility');
    await this._context.exposeBinding('_playwrightResume', false, () => {
      this._debugger.resume(false);
    }, 'main');
    await this._context.extendInjectedScript('utility', recorderSource.source, {
      isUnderTest: (0, _utils2.isUnderTest)()
    });
    await this._context.extendInjectedScript('main', consoleApiSource.source);
    if (this._debugger.isPaused()) this._pausedStateChanged();

    this._debugger.on(_debugger.Debugger.Events.PausedStateChanged, () => this._pausedStateChanged());

    this._context.recorderAppForTest = recorderApp;
  }

  _pausedStateChanged() {
    var _this$_recorderApp4;

    // If we are called upon page.pause, we don't have metadatas, populate them.
    for (const {
      metadata,
      sdkObject
    } of this._debugger.pausedDetails()) {
      if (!this._currentCallsMetadata.has(metadata)) this.onBeforeCall(sdkObject, metadata);
    }

    (_this$_recorderApp4 = this._recorderApp) === null || _this$_recorderApp4 === void 0 ? void 0 : _this$_recorderApp4.setPaused(this._debugger.isPaused());

    this._updateUserSources();

    this.updateCallLog([...this._currentCallsMetadata.keys()]);
  }

  _setMode(mode) {
    var _this$_recorderApp5;

    this._mode = mode;
    (_this$_recorderApp5 = this._recorderApp) === null || _this$_recorderApp5 === void 0 ? void 0 : _this$_recorderApp5.setMode(this._mode);

    this._generator.setEnabled(this._mode === 'recording');

    _debugger.Debugger.lookup(this._context).setMuted(this._mode === 'recording');

    if (this._mode !== 'none') this._context.pages()[0].bringToFront().catch(() => {});
  }

  _refreshOverlay() {
    for (const page of this._context.pages()) page.mainFrame().evaluateExpression('window._playwrightRefreshOverlay()', false, undefined, 'main').catch(() => {});
  }

  async _onPage(page) {
    // First page is called page, others are called popup1, popup2, etc.
    const frame = page.mainFrame();
    page.on('close', () => {
      this._pageAliases.delete(page);

      this._generator.addAction({
        pageAlias,
        ...(0, _utils.describeFrame)(page.mainFrame()),
        committed: true,
        action: {
          name: 'closePage',
          signals: []
        }
      });
    });
    frame.on(_frames.Frame.Events.Navigation, () => this._onFrameNavigated(frame, page));
    page.on(_page.Page.Events.Download, () => this._onDownload(page));
    page.on(_page.Page.Events.Dialog, () => this._onDialog(page));
    const suffix = this._pageAliases.size ? String(++this._lastPopupOrdinal) : '';
    const pageAlias = 'page' + suffix;

    this._pageAliases.set(page, pageAlias);

    if (page.opener()) {
      this._onPopup(page.opener(), page);
    } else {
      this._generator.addAction({
        pageAlias,
        ...(0, _utils.describeFrame)(page.mainFrame()),
        committed: true,
        action: {
          name: 'openPage',
          url: page.mainFrame().url(),
          signals: []
        }
      });
    }
  }

  _clearScript() {
    this._generator.restart();

    if (!!this._params.startRecording) {
      for (const page of this._context.pages()) this._onFrameNavigated(page.mainFrame(), page);
    }
  }

  async _performAction(frame, action) {
    // Commit last action so that no further signals are added to it.
    this._generator.commitLastAction();

    const page = frame._page;
    const actionInContext = {
      pageAlias: this._pageAliases.get(page),
      ...(0, _utils.describeFrame)(frame),
      action
    };

    this._generator.willPerformAction(actionInContext);

    const noCallMetadata = (0, _instrumentation.internalCallMetadata)();

    try {
      const kActionTimeout = 5000;

      if (action.name === 'click') {
        const {
          options
        } = (0, _utils.toClickOptions)(action);
        await frame.click(noCallMetadata, action.selector, { ...options,
          timeout: kActionTimeout
        });
      }

      if (action.name === 'press') {
        const modifiers = (0, _utils.toModifiers)(action.modifiers);
        const shortcut = [...modifiers, action.key].join('+');
        await frame.press(noCallMetadata, action.selector, shortcut, {
          timeout: kActionTimeout
        });
      }

      if (action.name === 'check') await frame.check(noCallMetadata, action.selector, {
        timeout: kActionTimeout
      });
      if (action.name === 'uncheck') await frame.uncheck(noCallMetadata, action.selector, {
        timeout: kActionTimeout
      });
      if (action.name === 'select') await frame.selectOption(noCallMetadata, action.selector, [], action.options.map(value => ({
        value
      })), {
        timeout: kActionTimeout
      });
    } catch (e) {
      this._generator.performedActionFailed(actionInContext);

      return;
    }

    const timer = setTimeout(() => {
      // Commit the action after 5 seconds so that no further signals are added to it.
      actionInContext.committed = true;

      this._timers.delete(timer);
    }, 5000);

    this._generator.didPerformAction(actionInContext);

    this._timers.add(timer);
  }

  async _recordAction(frame, action) {
    // Commit last action so that no further signals are added to it.
    this._generator.commitLastAction();

    this._generator.addAction({
      pageAlias: this._pageAliases.get(frame._page),
      ...(0, _utils.describeFrame)(frame),
      action
    });
  }

  _onFrameNavigated(frame, page) {
    const pageAlias = this._pageAliases.get(page);

    this._generator.signal(pageAlias, frame, {
      name: 'navigation',
      url: frame.url()
    });
  }

  _onPopup(page, popup) {
    const pageAlias = this._pageAliases.get(page);

    const popupAlias = this._pageAliases.get(popup);

    this._generator.signal(pageAlias, page.mainFrame(), {
      name: 'popup',
      popupAlias
    });
  }

  _onDownload(page) {
    const pageAlias = this._pageAliases.get(page);

    this._generator.signal(pageAlias, page.mainFrame(), {
      name: 'download',
      downloadAlias: String(++this._lastDownloadOrdinal)
    });
  }

  _onDialog(page) {
    const pageAlias = this._pageAliases.get(page);

    this._generator.signal(pageAlias, page.mainFrame(), {
      name: 'dialog',
      dialogAlias: String(++this._lastDialogOrdinal)
    });
  }

  async onBeforeCall(sdkObject, metadata) {
    if (this._mode === 'recording') return;

    this._currentCallsMetadata.set(metadata, sdkObject);

    this._allMetadatas.set(metadata.id, metadata);

    this._updateUserSources();

    this.updateCallLog([metadata]);

    if (metadata.params && metadata.params.selector) {
      var _this$_recorderApp6;

      this._highlightedSelector = metadata.params.selector;
      (_this$_recorderApp6 = this._recorderApp) === null || _this$_recorderApp6 === void 0 ? void 0 : _this$_recorderApp6.setSelector(this._highlightedSelector).catch(() => {});
    }
  }

  async onAfterCall(sdkObject, metadata) {
    if (this._mode === 'recording') return;
    if (!metadata.error) this._currentCallsMetadata.delete(metadata);

    this._updateUserSources();

    this.updateCallLog([metadata]);
  }

  _updateUserSources() {
    var _this$_recorderApp7;

    // Remove old decorations.
    for (const source of this._userSources.values()) {
      source.highlight = [];
      source.revealLine = undefined;
    } // Apply new decorations.


    let fileToSelect = undefined;

    for (const metadata of this._currentCallsMetadata.keys()) {
      if (!metadata.stack || !metadata.stack[0]) continue;
      const {
        file,
        line
      } = metadata.stack[0];

      let source = this._userSources.get(file);

      if (!source) {
        source = {
          file,
          text: this._readSource(file),
          highlight: [],
          language: languageForFile(file)
        };

        this._userSources.set(file, source);
      }

      if (line) {
        const paused = this._debugger.isPaused(metadata);

        source.highlight.push({
          line,
          type: metadata.error ? 'error' : paused ? 'paused' : 'running'
        });
        source.revealLine = line;
        fileToSelect = source.file;
      }
    }

    this._pushAllSources();

    if (fileToSelect) (_this$_recorderApp7 = this._recorderApp) === null || _this$_recorderApp7 === void 0 ? void 0 : _this$_recorderApp7.setFile(fileToSelect);
  }

  _pushAllSources() {
    var _this$_recorderApp8;

    (_this$_recorderApp8 = this._recorderApp) === null || _this$_recorderApp8 === void 0 ? void 0 : _this$_recorderApp8.setSources([...this._recorderSources, ...this._userSources.values()]);
  }

  async onBeforeInputAction(sdkObject, metadata) {}

  async onCallLog(logName, message, sdkObject, metadata) {
    this.updateCallLog([metadata]);
  }

  updateCallLog(metadatas) {
    var _this$_recorderApp9;

    if (this._mode === 'recording') return;
    const logs = [];

    for (const metadata of metadatas) {
      if (!metadata.method) continue;
      let status = 'done';
      if (this._currentCallsMetadata.has(metadata)) status = 'in-progress';
      if (this._debugger.isPaused(metadata)) status = 'paused';
      logs.push((0, _recorderUtils.metadataToCallLog)(metadata, status));
    }

    (_this$_recorderApp9 = this._recorderApp) === null || _this$_recorderApp9 === void 0 ? void 0 : _this$_recorderApp9.updateCallLogs(logs);
  }

  _readSource(fileName) {
    try {
      return fs.readFileSync(fileName, 'utf-8');
    } catch (e) {
      return '// No source available';
    }
  }

}

exports.RecorderSupplement = RecorderSupplement;

function languageForFile(file) {
  if (file.endsWith('.py')) return 'python';
  if (file.endsWith('.java')) return 'java';
  if (file.endsWith('.cs')) return 'csharp';
  return 'javascript';
}
//# sourceMappingURL=recorderSupplement.js.map