/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.documentdb;

import com.microsoft.azure.documentdb.DatabaseAccount;
import com.microsoft.azure.documentdb.DatabaseAccountLocation;
import com.microsoft.azure.documentdb.internal.DocumentServiceRequest;
import com.microsoft.azure.documentdb.internal.OperationType;
import com.microsoft.azure.documentdb.internal.ResourceType;
import com.microsoft.azure.documentdb.internal.Utils;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocationCache {
    private final Logger logger = LoggerFactory.getLogger(LocationCache.class);
    private final Collection<String> preferredLocations;
    private final boolean enableEndpointDiscovery;
    private final URI defaultEndpoint;
    private boolean enableMultipleWritableLocations;
    private boolean useMultipleWritableLocations;
    private final long backgroundRefreshLocationTimeIntervalInMS;
    private long lastCacheUpdateTimestamp;
    private DatabaseAccountLocationsInfo locationInfo;
    private ConcurrentMap<String, LocationUnavailabilityInfo> locationUnavailablityInfoByEndpoint;

    LocationCache(Collection<String> preferredLocations, URI defaultEndpoint, boolean enableEndpointDiscovery, boolean useMultipleWriteLocations, long backgroundRefreshLocationTimeIntervalInMS) {
        this.locationInfo = new DatabaseAccountLocationsInfo(defaultEndpoint);
        this.preferredLocations = preferredLocations;
        this.defaultEndpoint = defaultEndpoint;
        this.enableEndpointDiscovery = enableEndpointDiscovery;
        this.useMultipleWritableLocations = useMultipleWriteLocations;
        this.locationUnavailablityInfoByEndpoint = new ConcurrentHashMap<String, LocationUnavailabilityInfo>();
        this.backgroundRefreshLocationTimeIntervalInMS = backgroundRefreshLocationTimeIntervalInMS;
        this.lastCacheUpdateTimestamp = Long.MIN_VALUE;
    }

    List<URI> getWriteEndpoints() {
        if (this.locationUnavailablityInfoByEndpoint.size() > 0 && System.currentTimeMillis() - this.lastCacheUpdateTimestamp > this.backgroundRefreshLocationTimeIntervalInMS) {
            this.updateEndpointsCache();
        }
        return this.locationInfo.writeEndpoints;
    }

    List<URI> getReadEndpoints() {
        if (this.locationUnavailablityInfoByEndpoint.size() > 0 && System.currentTimeMillis() - this.lastCacheUpdateTimestamp > this.backgroundRefreshLocationTimeIntervalInMS) {
            this.updateEndpointsCache();
        }
        return this.locationInfo.readEndpoints;
    }

    URI getWriteEndpoint() {
        return this.getWriteEndpoints().get(0);
    }

    URI getReadEndpoint() {
        return this.getReadEndpoints().get(0);
    }

    List<String> getOrderedWriteEndpoints() {
        return this.locationInfo.availableWriteLocations;
    }

    List<String> getOrderedReadEndpoints() {
        return this.locationInfo.availableReadLocations;
    }

    void markCurrentLocationUnavailableForRead(URI endpoint) {
        this.markEndpointUnavailable(endpoint, EndpointOperationType.Read);
    }

    void markCurrentLocationUnavailableForWrite(URI endpoint) {
        this.markEndpointUnavailable(endpoint, EndpointOperationType.Write);
    }

    public void onDatabaseAccountRead(DatabaseAccount databaseAccount) {
        this.updateLocationCache(databaseAccount.getWritableLocations(), databaseAccount.getReadableLocations(), databaseAccount.getEnableMultipleWritableLocations());
    }

    URI resolveServiceEndpoint(DocumentServiceRequest request) {
        boolean usePreferredLocations;
        if (request.getLocationEndpointToRoute() != null) {
            return request.getLocationEndpointToRoute();
        }
        int locationIndex = request.getLocationIndexToRoute() != null ? request.getLocationIndexToRoute() : 0;
        boolean bl = usePreferredLocations = request.getUsePreferredLocations() != null ? request.getUsePreferredLocations() : true;
        if (!usePreferredLocations || Utils.isWriteOperation(request.getOperationType()) && !this.canUseMultipleWriteLocations(request)) {
            DatabaseAccountLocationsInfo currentLocationInfo = this.locationInfo;
            if (this.enableEndpointDiscovery && currentLocationInfo.availableWriteLocations.size() > 0) {
                locationIndex = Math.min(locationIndex % 2, currentLocationInfo.availableWriteLocations.size() - 1);
                String writeLocation = currentLocationInfo.availableWriteLocations.get(locationIndex);
                return currentLocationInfo.availableWriteEndpointByLocation.get(writeLocation);
            }
            return this.defaultEndpoint;
        }
        List<URI> endpoints = Utils.isWriteOperation(request.getOperationType()) ? this.getWriteEndpoints() : this.getReadEndpoints();
        return endpoints.get(locationIndex % endpoints.size());
    }

    boolean shouldRefreshEndpoints(CanRefreshInBackground canRefreshInBackground) {
        String mostPreferredLocation = this.preferredLocations != null && this.preferredLocations.size() > 0 ? this.preferredLocations.iterator().next() : null;
        canRefreshInBackground.setValue(true);
        DatabaseAccountLocationsInfo currentLocationInfo = this.locationInfo;
        if (this.enableEndpointDiscovery) {
            boolean shouldRefresh;
            boolean bl = shouldRefresh = this.useMultipleWritableLocations && !this.enableMultipleWritableLocations;
            if (StringUtils.isNotEmpty((CharSequence)mostPreferredLocation)) {
                if (currentLocationInfo.availableReadEndpointByLocation != null) {
                    URI mostPreferredReadEndpoint = currentLocationInfo.availableReadEndpointByLocation.get(mostPreferredLocation);
                    if (mostPreferredReadEndpoint != null && !mostPreferredReadEndpoint.equals(currentLocationInfo.readEndpoints.get(0))) {
                        this.logger.trace("shouldRefreshEndpoints = true since most preferred location [{}] is not available for read.", (Object)mostPreferredLocation);
                        return true;
                    }
                } else {
                    this.logger.trace("shouldRefreshEndpoints = true since most preferred location [{}] is not in available read locations.", (Object)mostPreferredLocation);
                    return true;
                }
            }
            if (!this.canUseMultipleWriteLocations()) {
                if (this.isEndpointUnavailable(currentLocationInfo.writeEndpoints.get(0), EndpointOperationType.Write)) {
                    canRefreshInBackground.setValue(currentLocationInfo.writeEndpoints.size() > 1);
                    this.logger.trace("shouldRefreshEndpoints = true since most preferred location [{}] endpoint [{}] is not available for write. canRefreshInBackground = [{}]", new Object[]{mostPreferredLocation, currentLocationInfo.writeEndpoints.get(0), canRefreshInBackground.getValue()});
                    return true;
                }
                return shouldRefresh;
            }
            if (StringUtils.isNotEmpty((CharSequence)mostPreferredLocation)) {
                URI mostPreferredWriteEndpoint = currentLocationInfo.availableWriteEndpointByLocation.get(mostPreferredLocation);
                if (mostPreferredWriteEndpoint != null) {
                    this.logger.trace("shouldRefreshEndpoints = [{}] since most preferred location [{}] is not available for write.", (Object)(shouldRefresh |= !mostPreferredWriteEndpoint.equals(currentLocationInfo.writeEndpoints.get(0))), (Object)mostPreferredLocation);
                    return shouldRefresh;
                }
                this.logger.trace("shouldRefreshEndpoints = true since most preferred location [{}] is not in available write locations", (Object)mostPreferredLocation);
                return true;
            }
            return shouldRefresh;
        }
        return false;
    }

    private void clearStaleEndpointUnavailabilityInfo() {
        if (this.locationUnavailablityInfoByEndpoint.size() > 0) {
            Set unavailableEndpoints = this.locationUnavailablityInfoByEndpoint.keySet();
            for (String unavailableEndpoint : unavailableEndpoints) {
                LocationUnavailabilityInfo unavailabilityInfo = (LocationUnavailabilityInfo)this.locationUnavailablityInfoByEndpoint.get(unavailableEndpoint);
                if (unavailabilityInfo == null || System.currentTimeMillis() - unavailabilityInfo.getLastUnavailabilityCheckTimeStamp() <= this.backgroundRefreshLocationTimeIntervalInMS) continue;
                this.locationUnavailablityInfoByEndpoint.remove(unavailableEndpoint);
                this.logger.trace("Removed endpoint [{}] unavailable for operations [{}] from unavailableEndpoints", (Object)unavailableEndpoint, (Object)unavailabilityInfo.getOperationTypes().toString());
            }
        }
    }

    private boolean isEndpointUnavailable(URI endpoint, EndpointOperationType expectedAvailableOperations) {
        LocationUnavailabilityInfo unavailabilityInfo = (LocationUnavailabilityInfo)this.locationUnavailablityInfoByEndpoint.get(endpoint.toString());
        if (expectedAvailableOperations == EndpointOperationType.None || unavailabilityInfo == null || !unavailabilityInfo.getOperationTypes().contains((Object)expectedAvailableOperations)) {
            return false;
        }
        if (System.currentTimeMillis() - unavailabilityInfo.getLastUnavailabilityCheckTimeStamp() > this.backgroundRefreshLocationTimeIntervalInMS) {
            return false;
        }
        this.logger.trace("Endpoint [{}] unavailable for operations [{}] present in unavailableEndpoints", (Object)endpoint, (Object)unavailabilityInfo.getOperationTypes().toString());
        return true;
    }

    private void markEndpointUnavailable(URI unavailableEndpoint, EndpointOperationType unavailableOperationType) {
        LocationUnavailabilityInfo unavailablilityInfo = (LocationUnavailabilityInfo)this.locationUnavailablityInfoByEndpoint.get(unavailableEndpoint.toString());
        Instant currentTime = Instant.now();
        if (unavailablilityInfo == null) {
            this.locationUnavailablityInfoByEndpoint.put(unavailableEndpoint.toString(), new LocationUnavailabilityInfo(currentTime.getMillis(), EnumSet.of(unavailableOperationType)));
        } else {
            EnumSet<EndpointOperationType> unavailableOperations = EnumSet.of(unavailableOperationType);
            unavailableOperations.addAll(unavailablilityInfo.getOperationTypes());
            this.locationUnavailablityInfoByEndpoint.put(unavailableEndpoint.toString(), new LocationUnavailabilityInfo(currentTime.getMillis(), unavailableOperations));
        }
        this.updateEndpointsCache();
        this.logger.trace("Endpoint [{}] unavailable for [{}] added/updated to unavailableEndpoints with timestamp [{}]", new Object[]{unavailableEndpoint, unavailableOperationType, currentTime.toDateTime()});
    }

    Collection<String> getPreferredLocations() {
        return this.preferredLocations;
    }

    private synchronized void updateEndpointsCache() {
        this.updateLocationCache(null, null, null);
    }

    synchronized void updateLocationCache(Iterable<DatabaseAccountLocation> writeLocations, Iterable<DatabaseAccountLocation> readLocations, Boolean enableMultipleWritableLocations) {
        DatabaseAccountLocationsInfo nextLocationInfo = new DatabaseAccountLocationsInfo(this.locationInfo);
        if (enableMultipleWritableLocations != null) {
            this.enableMultipleWritableLocations = enableMultipleWritableLocations;
        }
        this.clearStaleEndpointUnavailabilityInfo();
        if (this.enableEndpointDiscovery) {
            if (readLocations != null) {
                UpdatableList availableReadLocations = new UpdatableList(new ArrayList<String>());
                nextLocationInfo.availableReadEndpointByLocation = Collections.unmodifiableMap(this.getEndpointByLocation(readLocations, availableReadLocations));
                nextLocationInfo.availableReadLocations = availableReadLocations.getValue();
            }
            if (writeLocations != null) {
                UpdatableList availableWriteLocations = new UpdatableList(new ArrayList<String>());
                nextLocationInfo.availableWriteEndpointByLocation = Collections.unmodifiableMap(this.getEndpointByLocation(writeLocations, availableWriteLocations));
                nextLocationInfo.availableWriteLocations = availableWriteLocations.getValue();
            }
        }
        nextLocationInfo.writeEndpoints = this.GetPreferredAvailableEndpoints(nextLocationInfo.availableWriteEndpointByLocation, nextLocationInfo.availableWriteLocations, EndpointOperationType.Write, this.defaultEndpoint);
        nextLocationInfo.readEndpoints = this.GetPreferredAvailableEndpoints(nextLocationInfo.availableReadEndpointByLocation, nextLocationInfo.availableWriteLocations, EndpointOperationType.Read, nextLocationInfo.writeEndpoints.get(0));
        this.lastCacheUpdateTimestamp = System.currentTimeMillis();
        this.logger.trace("Current writeEndpoints = ({}) readEndpoints = ({})", (Object)String.join((CharSequence)", ", nextLocationInfo.writeEndpoints.toString()), (Object)String.join((CharSequence)", ", nextLocationInfo.readEndpoints.toString()));
        this.locationInfo = nextLocationInfo;
    }

    private List<URI> GetPreferredAvailableEndpoints(Map<String, URI> endpointsByLocation, List<String> orderedLocations, EndpointOperationType expectedAvailableOperation, URI fallbackEndpoint) {
        ArrayList<URI> endpoints = new ArrayList<URI>();
        if (this.enableEndpointDiscovery && endpointsByLocation != null && !endpointsByLocation.isEmpty()) {
            if (this.canUseMultipleWriteLocations() || expectedAvailableOperation == EndpointOperationType.Read) {
                ArrayList<URI> unavailableEndpoints = new ArrayList<URI>();
                if (this.preferredLocations != null && !this.preferredLocations.isEmpty()) {
                    for (String location : this.preferredLocations) {
                        URI endpoint = endpointsByLocation.get(location);
                        if (endpoint == null) continue;
                        if (this.isEndpointUnavailable(endpoint, expectedAvailableOperation)) {
                            unavailableEndpoints.add(endpoint);
                            continue;
                        }
                        endpoints.add(endpoint);
                    }
                }
                if (endpoints.size() == 0) {
                    endpoints.add(fallbackEndpoint);
                }
                endpoints.addAll(unavailableEndpoints);
            } else {
                for (String location : orderedLocations) {
                    URI endpoint;
                    if (StringUtils.isEmpty((CharSequence)location) || (endpoint = endpointsByLocation.get(location)) == null) continue;
                    endpoints.add(endpoint);
                }
            }
        }
        if (endpoints.size() == 0) {
            endpoints.add(fallbackEndpoint);
        }
        return Collections.unmodifiableList(endpoints);
    }

    Map<String, URI> getEndpointByLocation(Iterable<DatabaseAccountLocation> locations, UpdatableList orderedLocations) {
        LinkedHashMap<String, URI> endpointsByLocation = new LinkedHashMap<String, URI>();
        ArrayList<String> parsedLocations = new ArrayList<String>();
        for (DatabaseAccountLocation location : locations) {
            if (StringUtils.isEmpty((CharSequence)location.getName())) continue;
            try {
                URI regionUri = new URI(location.getEndpoint());
                endpointsByLocation.put(location.getName(), regionUri);
                parsedLocations.add(location.getName());
            }
            catch (URISyntaxException e) {
                this.logger.warn("GetAvailableEndpointsByLocation() - skipping add for location = [{}] as it is location name is either empty or endpoint is malformed [{}]", (Object)location.getName(), (Object)location.getEndpoint());
            }
        }
        orderedLocations.setValue(Collections.unmodifiableList(parsedLocations));
        return endpointsByLocation;
    }

    private boolean canUseMultipleWriteLocations() {
        return this.useMultipleWritableLocations && this.enableMultipleWritableLocations;
    }

    public boolean canUseMultipleWriteLocations(DocumentServiceRequest request) {
        return this.canUseMultipleWriteLocations() && (request.getResourceType() == ResourceType.Document || request.getResourceType() == ResourceType.StoredProcedure && request.getOperationType() == OperationType.ExecuteJavaScript);
    }

    private class DatabaseAccountLocationsInfo {
        public List<String> availableWriteLocations;
        public List<String> availableReadLocations;
        public Map<String, URI> availableReadEndpointByLocation;
        public Map<String, URI> availableWriteEndpointByLocation;
        public List<URI> writeEndpoints;
        public List<URI> readEndpoints;

        public DatabaseAccountLocationsInfo(URI defaultEndpoint) {
            this.availableReadLocations = new ArrayList<String>();
            this.availableWriteLocations = new ArrayList<String>();
            this.writeEndpoints = new ArrayList<URI>();
            this.writeEndpoints.add(defaultEndpoint);
            this.readEndpoints = new ArrayList<URI>();
            this.readEndpoints.add(defaultEndpoint);
        }

        public DatabaseAccountLocationsInfo(DatabaseAccountLocationsInfo other) {
            this.availableWriteLocations = new ArrayList<String>(other.availableWriteLocations);
            this.availableReadLocations = new ArrayList<String>(other.availableReadLocations);
            if (other.availableWriteEndpointByLocation != null) {
                this.availableWriteEndpointByLocation = new HashMap<String, URI>(other.availableWriteEndpointByLocation);
            }
            if (other.availableReadEndpointByLocation != null) {
                this.availableReadEndpointByLocation = new HashMap<String, URI>(other.availableReadEndpointByLocation);
            }
            this.writeEndpoints = new ArrayList<URI>(other.writeEndpoints);
            this.readEndpoints = new ArrayList<URI>(other.readEndpoints);
        }
    }

    public static class UpdatableList {
        private List<String> value;

        public UpdatableList(ArrayList<String> value) {
            this.value = new ArrayList<String>(value);
        }

        public List<String> getValue() {
            return this.value;
        }

        public void setValue(List<String> value) {
            this.value = new ArrayList<String>(value);
        }
    }

    public static class CanRefreshInBackground {
        private boolean value;

        public CanRefreshInBackground(boolean value) {
            this.value = value;
        }

        public boolean getValue() {
            return this.value;
        }

        public void setValue(boolean value) {
            this.value = value;
        }
    }

    private class LocationUnavailabilityInfo {
        private Long lastUnavailabilityCheckTimeStamp;
        private EnumSet<EndpointOperationType> operationTypes;

        public LocationUnavailabilityInfo(long timeStamp, EnumSet<EndpointOperationType> operationTypes) {
            this.lastUnavailabilityCheckTimeStamp = timeStamp;
            this.operationTypes = operationTypes;
        }

        public Long getLastUnavailabilityCheckTimeStamp() {
            return this.lastUnavailabilityCheckTimeStamp;
        }

        public EnumSet<EndpointOperationType> getOperationTypes() {
            return this.operationTypes;
        }
    }

    private static enum EndpointOperationType {
        None,
        Read,
        Write;

    }
}

