package com.microsoft.azure.documentdb.internal.query;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.microsoft.azure.documentdb.Document;
import com.microsoft.azure.documentdb.DocumentClientException;
import com.microsoft.azure.documentdb.DocumentQueryClientInternal;
import com.microsoft.azure.documentdb.FeedOptions;
import com.microsoft.azure.documentdb.SqlQuerySpec;
import com.microsoft.azure.documentdb.internal.query.executioncomponent.AggregateQueryExecutionComponent;
import com.microsoft.azure.documentdb.internal.query.executioncomponent.DefaultQueryExecutionComponent;
import com.microsoft.azure.documentdb.internal.query.executioncomponent.OrderByQueryExecutionComponent;
import com.microsoft.azure.documentdb.internal.query.executioncomponent.QueryExecutionComponent;
import com.microsoft.azure.documentdb.internal.query.executioncomponent.TopQueryExecutionComponent;

final class PipelinedQueryExecutionContext implements QueryExecutionContext<Document> {
    private static final int DEFAULT_PAGE_SIZE = 1000;
    private final int actualPageSize;
    protected final QueryExecutionComponent queryExecutionComponent;

    public PipelinedQueryExecutionContext(DocumentQueryClientInternal client,
            String collectionSelfLink,
            SqlQuerySpec querySpec, FeedOptions options,
            String resourceLink, PartitionedQueryExecutionInfo partitionedQueryExecutionInfo) {

        QueryExecutionContext queryExecutionContext;
        QueryInfo queryInfo = partitionedQueryExecutionInfo.getQueryInfo();
        Integer optionsPageSize = options.getPageSize();
        this.actualPageSize = (optionsPageSize != null && optionsPageSize > 0 ? optionsPageSize : DEFAULT_PAGE_SIZE);

        if (queryInfo.hasOrderBy()) {
            queryExecutionContext =  OrderByQueryExecutionContext.create(client, collectionSelfLink, querySpec,
                    options, resourceLink, partitionedQueryExecutionInfo);
        } else {
            queryExecutionContext = PeekingParallelDocumentQueryExecutionContext.create(client, collectionSelfLink, querySpec,
                    options, resourceLink, partitionedQueryExecutionInfo, actualPageSize);
        }

        QueryExecutionComponent currentQueryExecutionComponent = new DefaultQueryExecutionComponent(
                queryExecutionContext);

        if (queryInfo.hasAggregates()) {
            currentQueryExecutionComponent = new AggregateQueryExecutionComponent(currentQueryExecutionComponent,
                    queryInfo.getAggregates());
        }

        if (queryInfo.hasTop()) {
            currentQueryExecutionComponent = new TopQueryExecutionComponent(currentQueryExecutionComponent,
                    queryInfo.getTop());
        }

        if (queryInfo.hasOrderBy()) {
            currentQueryExecutionComponent = new OrderByQueryExecutionComponent(currentQueryExecutionComponent);
        }

        this.queryExecutionComponent = currentQueryExecutionComponent;


    }

    @Override
    public List<Document> fetchNextBlock() throws DocumentClientException {
        List<Document> result = null;
        while ((result == null || result.size() < this.actualPageSize) && this.hasNext()) {
            if (result == null) {
                result = new ArrayList<Document>(this.actualPageSize);
            }

            Document document = this.next();
            if (document != null) {
                result.add(document);
            }
        }

        return result;
    }

    @Override
    public boolean hasNext() {
        return this.queryExecutionComponent.hasNext();
    }

    @Override
    public Document next() {
        return this.queryExecutionComponent.next();
    }

    @Override
    public void remove() {
        this.queryExecutionComponent.remove();
    }

    @Override
    public Map<String, String> getResponseHeaders() {
        return this.queryExecutionComponent.getResponseHeaders();
    }

    @Override
    public void onNotifyStop() {
        this.queryExecutionComponent.onNotifyStop();
    }
}