/*
 * Copyright (c) Microsoft Corporation.  All rights reserved.
 */

package com.microsoft.azure.documentdb.internal;

import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.microsoft.azure.documentdb.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import org.json.JSONArray;
import org.json.JSONObject;

import com.microsoft.azure.documentdb.internal.directconnectivity.Address;
import com.microsoft.azure.documentdb.internal.directconnectivity.StoreResponse;

/**
 * This is core Transport/Connection agnostic response for the Azure Cosmos DB database service.
 */
public class DocumentServiceResponse {
    private int statusCode;
    private Map<String, String> headersMap = new HashMap<String, String>();
    private StoreResponse storeResponse;

    public DocumentServiceResponse(HttpResponse httpResponse, boolean isMedia, ClientSideRequestStatistics requestStatistics) {
        this(StoreResponse.fromHttpResponse(httpResponse, isMedia, requestStatistics));
    }

    public DocumentServiceResponse(StoreResponse response) {
        // Gets status code.
        this.statusCode = response.getStatus();

        // TODO: handle session token

        // Extracts headers.
        for (int i = 0; i < response.getResponseHeaderNames().length; i++) {
            if (!response.getResponseHeaderNames()[i].equals(HttpConstants.HttpHeaders.LSN)) {
                this.headersMap.put(response.getResponseHeaderNames()[i], response.getResponseHeaderValues()[i]);
            }
        }

        this.storeResponse = response;
    }

    public static <T extends Resource> String getResourceKey(Class<T> c) {
        if (c.equals(Attachment.class)) {
            return InternalConstants.ResourceKeys.ATTACHMENTS;
        } else if (c.equals(Conflict.class)) {
            return InternalConstants.ResourceKeys.CONFLICTS;
        } else if (c.equals(Database.class)) {
            return InternalConstants.ResourceKeys.DATABASES;
        } else if (Document.class.isAssignableFrom(c)) {
            return InternalConstants.ResourceKeys.DOCUMENTS;
        } else if (c.equals(DocumentCollection.class)) {
            return InternalConstants.ResourceKeys.DOCUMENT_COLLECTIONS;
        } else if (c.equals(Offer.class)) {
            return InternalConstants.ResourceKeys.OFFERS;
        } else if (c.equals(Permission.class)) {
            return InternalConstants.ResourceKeys.PERMISSIONS;
        } else if (c.equals(Trigger.class)) {
            return InternalConstants.ResourceKeys.TRIGGERS;
        } else if (c.equals(StoredProcedure.class)) {
            return InternalConstants.ResourceKeys.STOREDPROCEDURES;
        } else if (c.equals(User.class)) {
            return InternalConstants.ResourceKeys.USERS;
        } else if (c.equals(UserDefinedFunction.class)) {
            return InternalConstants.ResourceKeys.USER_DEFINED_FUNCTIONS;
        } else if (c.equals(Address.class)) {
            return InternalConstants.ResourceKeys.ADDRESSES;
        } else if (c.equals(PartitionKeyRange.class)) {
            return InternalConstants.ResourceKeys.PARTITION_KEY_RANGES;
        }

        throw new IllegalArgumentException("c");
    }

    public int getStatusCode() {
        return this.statusCode;
    }

    public Map<String, String> getResponseHeaders() {
        return this.headersMap;
    }

    public String getReponseBodyAsString() {
        return this.storeResponse.getResponseBody();
    }

    public <T extends Resource> T getResource(Class<T> c) {
        String responseBody = this.getReponseBodyAsString();
        if (StringUtils.isEmpty(responseBody))
            return null;

        try {
            return c.getConstructor(String.class).newInstance(responseBody);
        } catch (InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException
                | NoSuchMethodException | SecurityException e) {
            throw new IllegalStateException("Failed to instantiate class object.", e);
        }
    }

    public <T extends Resource> List<T> getQueryResponse(Class<T> c) {
        String responseBody = this.getReponseBodyAsString();
        if (responseBody == null) {
            return new ArrayList<T>();
        }

        JSONObject jobject = new JSONObject(responseBody);
        String resourceKey = DocumentServiceResponse.getResourceKey(c);
        JSONArray jTokenArray = jobject.getJSONArray(resourceKey);

        // Aggregate queries may return a nested array
        JSONArray innerArray;
        while (jTokenArray != null && jTokenArray.length() == 1 && (innerArray = jTokenArray.optJSONArray(0)) != null) {
            jTokenArray = innerArray;
        }

        List<T> queryResults = new ArrayList<T>();

        if (jTokenArray != null) {
            for (int i = 0; i < jTokenArray.length(); ++i) {
                Object jToken = jTokenArray.get(i);
                // Aggregate on single partition collection may return the aggregated value only
                // In that case it needs to encapsulated in a special document
                String resourceJson = jToken instanceof Number || jToken instanceof Boolean
                        ? String.format("{\"%s\": %s}", Constants.Properties.AGGREGATE, jToken.toString())
                                : jToken.toString();
                        T resource = null;
                        try {
                            resource = c.getConstructor(String.class).newInstance(resourceJson);
                        } catch (InstantiationException | IllegalAccessException | IllegalArgumentException
                                | InvocationTargetException | NoSuchMethodException | SecurityException e) {
                            throw new IllegalStateException("Failed to instantiate class object.", e);
                        }

                        queryResults.add(resource);
            }
        }

        return queryResults;
    }

    public InputStream getContentStream() {
        return this.storeResponse.getResponseStream();
    }

    public ClientSideRequestStatistics getClientSideRequestStatistics() {
        if (this.storeResponse == null) {
            return null;
        }
        return this.storeResponse.getClientSideRequestStatistics();
    }
}
