/*
 * Copyright (c) Microsoft Corporation.  All rights reserved.
 */

package com.microsoft.azure.documentdb;

/**
 * Specifies the options associated with feed methods (enumeration operations) in the Azure Cosmos DB database service.
 */
public final class FeedOptions extends FeedOptionsBase {
    private String sessionToken;
    private String partitionKeyRangeId;
    private Boolean enableScanInQuery;
    private Boolean emitVerboseTracesInQuery;
    private PartitionKey partitionkey;
    private Boolean enableCrossPartitionQuery;
    private int maxDegreeOfParallelism;
    private int maxBufferedItemCount;
    private Boolean disableRUPerMinuteUsage;
    private int responseContinuationTokenLimitInKb = 0;

    public FeedOptions() {}

    public FeedOptions(FeedOptions options) {
        super(options);
        this.sessionToken = options.sessionToken;
        this.partitionKeyRangeId = options.partitionKeyRangeId;
        this.enableScanInQuery = options.enableScanInQuery;
        this.emitVerboseTracesInQuery = options.emitVerboseTracesInQuery;
        this.partitionkey = options.partitionkey;
        this.enableCrossPartitionQuery = options.enableCrossPartitionQuery;
        this.maxDegreeOfParallelism = options.maxDegreeOfParallelism;
        this.maxBufferedItemCount = options.maxBufferedItemCount;
        this.disableRUPerMinuteUsage = options.disableRUPerMinuteUsage;
        this.responseContinuationTokenLimitInKb = options.responseContinuationTokenLimitInKb;
    }

    /**
     * Gets the partitionKeyRangeId.
     *
     * @return the partitionKeyRangeId.
     */
    public String getPartitionKeyRangeIdInternal () {
        return this.partitionKeyRangeId;
    }

    /**
     * Sets the partitionKeyRangeId.
     *
     * @param partitionKeyRangeId
     *           the partitionKeyRangeId.
     */
    public void setPartitionKeyRangeIdInternal(String partitionKeyRangeId) {
        this.partitionKeyRangeId = partitionKeyRangeId;
    }

    /**
     * Gets the session token for use with session consistency.
     *
     * @return the session token.
     */
    public String getSessionToken() {
        return this.sessionToken;
    }

    /**
     * Sets the session token for use with session consistency.
     *
     * @param sessionToken
     *            the session token.
     */
    public void setSessionToken(String sessionToken) {
        this.sessionToken = sessionToken;
    }

    /**
     * Gets the option to allow scan on the queries which couldn't be served as
     * indexing was opted out on the requested paths.
     *
     * @return the option of enable scan in query.
     */
    public Boolean getEnableScanInQuery() {
        return this.enableScanInQuery;
    }

    /**
     * Sets the option to allow scan on the queries which couldn't be served as
     * indexing was opted out on the requested paths.
     *
     * @param enableScanInQuery
     *            the option of enable scan in query.
     */
    public void setEnableScanInQuery(Boolean enableScanInQuery) {
        this.enableScanInQuery = enableScanInQuery;
    }

    /**
     * Gets the option to allow queries to emit out verbose traces for
     * investigation.
     *
     * @return the emit verbose traces in query.
     */
    public Boolean getEmitVerboseTracesInQuery() {
        return this.emitVerboseTracesInQuery;
    }

    /**
     * Sets the option to allow queries to emit out verbose traces for
     * investigation.
     *
     * @param emitVerboseTracesInQuery
     *            the emit verbose traces in query.
     */
    public void setEmitVerboseTracesInQuery(Boolean emitVerboseTracesInQuery) {
        this.emitVerboseTracesInQuery = emitVerboseTracesInQuery;
    }

    /**
     * Gets the partition key used to identify the current request's target
     * partition.
     *
     * @return the partition key.
     */
    public PartitionKey getPartitionKey() {
        return this.partitionkey;
    }

    /**
     * Sets the partition key used to identify the current request's target
     * partition.
     *
     * @param partitionkey
     *            the partition key value.
     */
    public void setPartitionKey(PartitionKey partitionkey) {
        this.partitionkey = partitionkey;
    }

    /**
     * Gets the option to allow queries to run across all partitions of the
     * collection.
     *
     * @return whether to allow queries to run across all partitions of the
     *         collection.
     */
    public Boolean getEnableCrossPartitionQuery() {
        return this.enableCrossPartitionQuery;
    }

    /**
     * Sets the option to allow queries to run across all partitions of the
     * collection.
     *
     * @param enableCrossPartitionQuery
     *            whether to allow queries to run across all partitions of the
     *            collection.
     */
    public void setEnableCrossPartitionQuery(Boolean enableCrossPartitionQuery) {
        this.enableCrossPartitionQuery = enableCrossPartitionQuery;
    }

    /**
     * Gets the number of concurrent operations run client side during parallel
     * query execution.
     *
     * @return number of concurrent operations run client side during parallel
     *         query execution.
     */
    public int getMaxDegreeOfParallelism() {
        return maxDegreeOfParallelism;
    }

    /**
     * Sets the number of concurrent operations run client side during parallel
     * query execution.
     *
     * @param maxDegreeOfParallelism
     *            number of concurrent operations.
     */
    public void setMaxDegreeOfParallelism(int maxDegreeOfParallelism) {
        this.maxDegreeOfParallelism = maxDegreeOfParallelism;
    }

    /**
     * Gets the maximum number of items that can be buffered client side during
     * parallel query execution.
     *
     * @return maximum number of items that can be buffered client side during
     *         parallel query execution.
     */
    public int getMaxBufferedItemCount() {
        return maxBufferedItemCount;
    }

    /**
     * Sets the maximum number of items that can be buffered client side during
     * parallel query execution.
     *
     * @param maxBufferedItemCount
     *            maximum number of items.
     */
    public void setMaxBufferedItemCount(int maxBufferedItemCount) {
        this.maxBufferedItemCount = maxBufferedItemCount;
    }

    /**
     * Sets the disableRUPerMinuteUsage option for the current query in the
     * Azure Cosmos DB database service.
     *
     * disableRUPerMinuteUsage is used to enable/disable Request
     * Units(RUs)/minute capacity to serve the query if regular provisioned
     * RUs/second is exhausted.
     *
     * @param disableRUPerMinuteUsage
     *            used to set disableRUPerMinuteUsage
     */
    public void setDisableRUPerMinuteUsage(boolean disableRUPerMinuteUsage) {
        this.disableRUPerMinuteUsage = disableRUPerMinuteUsage;
    }

    /**
     * Gets disableRUPerMinuteUsage option for the current query in the Azure Cosmos DB database service.
     * @return true if disableRUPerMinuteUsage is set; otherwise false
     */
    public Boolean getDisableRUPerMinuteUsage() {
        return disableRUPerMinuteUsage;
    }

    /**
     * Sets the ResponseContinuationTokenLimitInKb" request option for document query requests
     * in the Azure Cosmos DB service.
     *
     * ResponseContinuationTokenLimitInKb is used to limit the length of continuation token in the query response.
     * Valid values are &gt;= 1.
     *
     * The continuation token contains both required and optional fields.
     * The required fields are necessary for resuming the execution from where it was stooped.
     * The optional fields may contain serialized index lookup work that was done but not yet utilized.
     * This avoids redoing the work again in subsequent continuations and hence improve the query performance.
     * Setting the maximum continuation size to 1KB, the Azure Cosmos DB service will only serialize required fields.
     * Starting from 2KB, the Azure Cosmos DB service would serialize as much as it could fit till it reaches the maximum specified size.
     *
     *  @param limitInKb continuation token size limit.
     */
    public void setResponseContinuationTokenLimitInKb(int limitInKb) {
        this.responseContinuationTokenLimitInKb = limitInKb;
    }

    /**
     * Gets the ResponseContinuationTokenLimitInKb request option for document query requests
     * in the Azure Cosmos DB service. If not already set returns 0.
     *
     * ResponseContinuationTokenLimitInKb is used to limit the length of continuation token in the query response.
     * Valid values are &gt;= 1.
     *
     * @return return set ResponseContinuationTokenLimitInKb, or 0 if not set
     */
    public int getResponseContinuationTokenLimitInKb() {
        return responseContinuationTokenLimitInKb;
    }
}
