package com.microsoft.azure.documentdb;

import java.lang.reflect.InvocationTargetException;

import org.json.JSONObject;

import com.microsoft.azure.documentdb.internal.Constants;

/**
 * Represents a conflict in the version of a particular resource in the Azure Cosmos DB database service.
 * <p>
 * During rare failure scenarios, conflicts are generated for the documents in transit. Clients can inspect the
 * respective conflict instances  for resources and operations in conflict.
 */
public final class Conflict extends Resource {
    /**
     * Initialize a conflict object.
     */
    public Conflict() {
        super();
    }

    /**
     * Initialize a conflict object from json string.
     *
     * @param jsonString the json string that represents the conflict.
     */
    public Conflict(String jsonString) {
        super(jsonString);
    }

    /**
     * Initialize a conflict object from json object.
     *
     * @param jsonObject the json object that represents the conflict.
     */
    public Conflict(JSONObject jsonObject) {
        super(jsonObject);
    }

    /**
     * Gets the operation kind.
     *
     * @return the operation kind.
     */
    public OperationKind getOperationKind() {
        try {
            return OperationKind.valueOf(super.getString(Constants.Properties.OPERATION_TYPE));
        } catch (IllegalArgumentException e) {
            return OperationKind.invalid;
        }
    }

    /**
     * Gets the type of the conflicting resource.
     *
     * @return the resource type.
     */
    public String getResourceType() {
        return super.getString(Constants.Properties.RESOURCE_TYPE);
    }
    
    /**
     * Gets the ID associated with the resource.
     *
     * @return the ID associated with the resource.
     */
    public String getSourceResourceId() {
        return super.getString(Constants.Properties.SOURCE_RESOURCE_ID);
    }

    /**
     * Gets the conflicting resource in the Azure Cosmos DB service.
     *
     * @param <T>           The type of the object.
     * @param resourceClass The returned type of conflicting resource.
     * 
     * @return The conflicting resource.
     */
    public <T extends Resource> T getResource(Class<T> resourceClass) {
        String content = super.getString(Constants.Properties.CONTENT);
        try {
            return resourceClass.getConstructor(String.class).newInstance(content);
        } catch (InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException
                | NoSuchMethodException | SecurityException e) {
            throw new IllegalStateException("Failed to create resource.", e);
        }
    }
}
