package com.microsoft.azure.documentdb;

import org.joda.time.DateTime;

/**
 * Specifies the options associated with change feed methods (enumeration
 * operations) in the Azure Cosmos DB database service.
 */
public final class ChangeFeedOptions extends FeedOptionsBase {
    private String partitionKeyRangeId;
    private boolean startFromBeginning;
    private DateTime startDateTime;

    /**
     * Get the partition key range id for the current request
     * <p>
     * ChangeFeed requests can be executed against specific partition key ranges.
     * This is used to process the change feed in parallel across multiple consumers.
     * </p>
     *
     * @return a string indicating the partition key range ID
     * @see PartitionKeyRange
     * @see DocumentClient#readPartitionKeyRanges(String, FeedOptions)
     */
    public String getPartitionKeyRangeId() {
        return partitionKeyRangeId;
    }

    /**
     * Set the partition key range id for the current request
     * <p>
     * ChangeFeed requests can be executed against specific partition key ranges.
     * This is used to process the change feed in parallel across multiple consumers.
     * </p>
     *
     * @param partitionKeyRangeId a string indicating the partition key range ID
     * @see PartitionKeyRange
     * @see DocumentClient#readPartitionKeyRanges(String, FeedOptions)
     */
    public void setPartitionKeyRangeId(String partitionKeyRangeId) {
        this.partitionKeyRangeId = partitionKeyRangeId;
    }

    /**
     * Get whether change feed should start from beginning (true) or from current (false).
     * By default it's start from current (false).
     *
     * @return a boolean value indicating change feed should start from beginning or not
     */
    public boolean isStartFromBeginning() {
        return startFromBeginning;
    }

    /**
     * Set whether change feed should start from beginning (true) or from current (false).
     * By default it's start from current (false).
     *
     * @param startFromBeginning a boolean value indicating change feed should start from beginning or not
     */
    public void setStartFromBeginning(boolean startFromBeginning) {
        this.startFromBeginning = startFromBeginning;
    }

    /**
     * Gets the UTC date time to start looking for changes after.
     * @return a UTC date time to start looking for changes after, if set or null otherwise
     */
    public DateTime getStartDateTime() {
        return this.startDateTime;
    }

    /**
     * Sets the UTC date time (exclusive) to start looking for changes after.
     * If this is specified, startFromBeginning is ignored. 
     * @param startDateTime a UTC date time to start looking for changes after.
     */
    public void setStartDateTime(DateTime startDateTime) {
        this.startDateTime = startDateTime;
    }
}
