/**
 * Copyright (c) 2000-2022 Liferay, Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.liferay.faces.bridge.internal;

import java.io.IOException;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.render.ResponseStateManager;
import javax.portlet.BaseURL;
import javax.portlet.MimeResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletRequest;
import javax.portlet.PortletURL;
import javax.portlet.ResourceURL;
import javax.portlet.faces.Bridge;
import javax.portlet.faces.BridgeConfig;
import javax.portlet.faces.BridgeURL;
import javax.portlet.faces.BridgeUtil;
import javax.portlet.filter.PortletURLWrapper;

import com.liferay.faces.util.config.ConfiguredServletMapping;
import com.liferay.faces.util.logging.Logger;
import com.liferay.faces.util.logging.LoggerFactory;
import com.liferay.faces.util.render.FacesURLEncoder;


/**
 * @author  Neil Griffin
 */
public abstract class BridgeURLBase extends BridgeURLBaseCompat implements BridgeURL {

	// Logger
	private static final Logger logger = LoggerFactory.getLogger(BridgeURLBase.class);

	// Protected Data Members
	protected BridgeURI bridgeURI;
	protected List<ConfiguredServletMapping> configuredFacesServletMappings;
	protected String contextPath;
	protected String currentViewId;
	protected boolean selfReferencing;
	protected String viewIdResourceParameterName;
	protected String viewIdRenderParameterName;

	// Private Data Members
	private PortletConfig portletConfig;
	private String viewId;

	@SuppressWarnings("unchecked")
	public BridgeURLBase(String uri, String contextPath, String namespace, String encoding,
		FacesURLEncoder facesURLEncoder, String currentViewId, PortletConfig portletConfig, BridgeConfig bridgeConfig)
		throws URISyntaxException, UnsupportedEncodingException {

		this.bridgeURI = new BridgeURI(uri, namespace, facesURLEncoder, encoding);
		this.configuredFacesServletMappings = (List<ConfiguredServletMapping>) bridgeConfig.getAttributes().get(
				BridgeConfigAttributeMap.CONFIGURED_FACES_SERVLET_MAPPINGS);
		this.contextPath = contextPath;
		this.currentViewId = currentViewId;
		this.portletConfig = portletConfig;
		this.viewIdRenderParameterName = bridgeConfig.getViewIdRenderParameterName();
		this.viewIdResourceParameterName = bridgeConfig.getViewIdResourceParameterName();
	}

	@Override
	public String getParameter(String name) {
		return bridgeURI.getParameter(name);
	}

	@Override
	public Map<String, String[]> getParameterMap() {
		return bridgeURI.getParameterMap();
	}

	@Override
	public String getViewId() {

		if (viewId == null) {

			String contextRelativePath = bridgeURI.getContextRelativePath(contextPath);

			if ((currentViewId != null) && (currentViewId.equals(contextRelativePath))) {
				viewId = currentViewId;
			}
			else {

				// If the context relative path maps to an actual Faces View due to an implicit servlet mapping or an
				// explicit servlet-mapping entry in the WEB-INF/web.xml descriptor, then the context relative path
				// is a faces view target.
				if (isViewPathMappedToFacesServlet(contextRelativePath)) {
					viewId = contextRelativePath;
				}

				// Otherwise,
				else {
					String potentialFacesViewId;

					// If the context relative path is not available, then
					if (contextRelativePath == null) {

						// TCK (modeViewIDTest)
						// * currentViewId="/tests/modeViewIdTest.xhtml"
						//
						// TCK (requestRenderIgnoresScopeViaCreateViewTest)
						// TCK (requestRenderRedisplayTest)
						// TCK (requestRedisplayOutOfScopeTest)
						// TCK (renderRedirectTest)
						// TCK (ignoreCurrentViewIdModeChangeTest)
						// TCK (exceptionThrownWhenNoDefaultViewIdTest)
						// * currentViewId="/tests/redisplayRenderNewModeRequestTest.xhtml"
						//
						// TCK (resourceAttrRetainedAfterRedisplayPPRTest)
						// * currentViewId="/tests/redisplayResourceAjaxResult.xhtml"
						//
						// TCK (encodeActionURLPortletRenderTest)
						// TCK (encodeActionURLPortletActionTest)
						// * currentViewId="/tests/singleRequestTest.xhtml"
						//
						// TCK (redirectRenderPRP1Test)
						// * currentViewId=null
						potentialFacesViewId = currentViewId;
					}

					// Otherwise, if the context relative path is indeed available, then
					else {

						// TCK (headerOrRenderPhaseListenerTest)
						// TCK (encodeActionURLWithWindowStateActionTest)
						// TCK (encodeActionURLNonJSFViewRenderTest)
						// TCK (encodeActionURLNonJSFViewWithParamRenderTest)
						// TCK (encodeActionURLNonJSFViewWithModeRenderTest)
						// TCK (encodeActionURLNonJSFViewWithInvalidModeRenderTest)
						// TCK (encodeActionURLNonJSFViewWithWindowStateRenderTest)
						// TCK (encodeActionURLNonJSFViewWithInvalidWindowStateRenderTest)
						// TCK (encodeActionURLNonJSFViewResourceTest)
						// TCK (encodeActionURLNonJSFViewWithParamResourceTest)
						// TCK (encodeActionURLNonJSFViewWithModeResourceTest)
						// TCK (encodeActionURLNonJSFViewWithInvalidModeResourceTest)
						// TCK (encodeActionURLNonJSFViewWithWindowStateResourceTest)
						// TCK (encodeActionURLNonJSFViewWithInvalidWindowStateResourceTest)
						// * contextRelativeViewPath="/nonFacesViewTestPortlet.ptlt"
						// * currentViewId="/tests/nonJSFViewTest.xhtml"
						//
						// TCK (nonFacesResourceTest)
						// * contextRelativeViewPath="/tck/nonFacesResource"
						// * currentViewId="/tests/nonFacesResourceTest.xhtml"
						//
						// TCK (encodeResourceURLBackLinkTest)
						// * contextRelativePath="/resources/myImage.jpg"
						// * currentViewId="/tests/singleRequestTest.xhtml"
						//
						// TCK (illegalRedirectRenderTest)
						// * contextRelativeViewPath="/tests/NonJSFView.portlet"
						// * currentViewId=null
						potentialFacesViewId = contextRelativePath;
					}

					// If the extension/suffix of the context relative path matches that of the current viewId at the
					// time of construction, then it is a it is a faces view target.
					if ((currentViewId != null) && (matchPathAndExtension(currentViewId, potentialFacesViewId))) {

						// TCK (modeViewIDTest)
						// * contextRelativeViewPath=null
						// * potentialFacesViewId="/tests/modeViewIdTest.xhtml"
						// * currentViewId="/tests/modeViewIdTest.xhtml"
						viewId = potentialFacesViewId;

						logger.debug(
							"Regarding path=[{0}] as a Faces view since it has the same path and extension as the current viewId=[{1}]",
							potentialFacesViewId, currentViewId);
					}
				}
			}
		}

		return viewId;
	}

	@Override
	public String removeParameter(String name) {
		return bridgeURI.removeParameter(name);
	}

	@Override
	public void setParameter(String name, String[] value) {
		bridgeURI.setParameter(name, value);
	}

	@Override
	public void setParameter(String name, String value) {
		bridgeURI.setParameter(name, new String[] { value });
	}

	@Override
	public String toString() {

		String stringValue = null;

		try {

			// Ask the Portlet Container for a BaseURL that contains the modified parameters.
			FacesContext facesContext = FacesContext.getCurrentInstance();
			BaseURL baseURL = toBaseURL(facesContext);

			// If the URL string has escaped characters (like "&amp;" for "&", etc) then ask the
			// portlet container to create an escaped representation of the URL string.
			if (bridgeURI.isEscaped()) {

				StringWriter urlWriter = new StringWriter();

				try {

					baseURL.write(urlWriter, true);
					stringValue = urlWriter.toString();
				}
				catch (IOException e) {

					logger.error(e);
					stringValue = baseURL.toString();
				}
			}

			// Otherwise, ask the portlet container to create a normal (non-escaped) string
			// representation of the URL string.
			else {
				stringValue = baseURL.toString();
			}

			// FACES-2978: In order to support a redirect that occurs during the RENDER_PHASE of the portlet lifecycle
			// as an "in-place" navigation (a.k.a. "render-redirect"), the viewId is saved as a request attribute that
			// is retrieved by ExternalContext.redirect(String url). This is necessary because the redirect URL passed
			// to ExternalContext.redirect(String url) is already encoded by ExternalContext.encodeActionURL(String url)
			// and the bridge does not have the ability to parse the encoded redirect URL for the target viewId
			// parameter.
			if (baseURL instanceof FacesViewActionURL) {

				FacesViewActionURL facesViewActionURL = (FacesViewActionURL) baseURL;
				String viewId = facesViewActionURL.getViewId();

				if (viewId != null) {
					ExternalContext externalContext = facesContext.getExternalContext();
					Map<String, Object> requestMap = externalContext.getRequestMap();
					String requestMapKey = Bridge.VIEW_ID + stringValue;
					requestMap.put(requestMapKey, viewId);
				}
			}
		}
		catch (MalformedURLException e) {
			logger.error(e);
		}

		return stringValue;
	}

	protected abstract BaseURL toBaseURL(FacesContext facesContext) throws MalformedURLException;

	protected void copyRenderParameters(PortletRequest portletRequest, BaseURL baseURL, String namespace,
		char separatorChar) {

		// Copy the public render parameters of the current view to the BaseURL.
		Map<String, String[]> publicParameterMap = portletRequest.getPublicParameterMap();

		if (publicParameterMap != null) {
			copyParameterMapToBaseURL(publicParameterMap, baseURL, namespace, separatorChar);
		}

		// Copy the private render parameters of the current view to the BaseURL.
		Map<String, String[]> privateParameterMap = portletRequest.getPrivateParameterMap();

		if (privateParameterMap != null) {
			copyParameterMapToBaseURL(privateParameterMap, baseURL, namespace, separatorChar);
		}
	}

	protected PortletURL createActionURL(FacesContext facesContext, boolean modeChanged) throws MalformedURLException {

		List<URIParameter> toStringParameters = getToStringParameters(modeChanged);

		return createActionURL(facesContext, toStringParameters);
	}

	protected PortletURL createActionURL(FacesContext facesContext, Set<String> excludedParameterNames)
		throws MalformedURLException {

		List<URIParameter> toStringParameters = getToStringParameters(false, excludedParameterNames);

		return createActionURL(facesContext, toStringParameters);
	}

	protected PortletURL createRenderURL(FacesContext facesContext, Map<String, String[]> parameterMap)
		throws MalformedURLException {

		List<URIParameter> uriParameters = parameterMapToList(parameterMap);

		return createRenderURL(facesContext, uriParameters);
	}

	protected PortletURL createRenderURL(FacesContext facesContext, Set<String> excludedParameterNames)
		throws MalformedURLException {

		List<URIParameter> toStringParameters = getToStringParameters(false, excludedParameterNames);

		return createRenderURL(facesContext, toStringParameters);
	}

	protected PortletURL createRenderURL(FacesContext facesContext, boolean modeChanged) throws MalformedURLException {

		List<URIParameter> toStringParameters = getToStringParameters(modeChanged);

		return createRenderURL(facesContext, toStringParameters);
	}

	protected ResourceURL createResourceURL(FacesContext facesContext) throws MalformedURLException {
		return createResourceURL(facesContext, (List<URIParameter>) null);
	}

	protected ResourceURL createResourceURL(FacesContext facesContext, Map<String, String[]> parameterMap)
		throws MalformedURLException {

		List<URIParameter> uriParameters = getToStringParameters(false);
		Set<Map.Entry<String, String[]>> entrySet = parameterMap.entrySet();

		for (Map.Entry<String, String[]> mapEntry : entrySet) {
			uriParameters.add(new URIParameter(mapEntry.getKey(), mapEntry.getValue()));
		}

		return createResourceURL(facesContext, uriParameters);
	}

	protected ResourceURL createResourceURL(FacesContext facesContext, Set<String> excludedParameterNames)
		throws MalformedURLException {

		List<URIParameter> toStringParameters = getToStringParameters(false, excludedParameterNames);

		return createResourceURL(facesContext, toStringParameters);
	}

	protected ResourceURL createResourceURL(FacesContext facesContext, boolean modeChanged)
		throws MalformedURLException {

		List<URIParameter> toStringParameters = getToStringParameters(modeChanged);

		return createResourceURL(facesContext, toStringParameters);
	}

	protected String getViewIdParameterName() {

		if (bridgeURI.isPortletScheme() && (bridgeURI.getPortletPhase() == Bridge.PortletPhase.RESOURCE_PHASE)) {
			return viewIdResourceParameterName;
		}
		else {
			return viewIdRenderParameterName;
		}
	}

	private void copyParameterMapToBaseURL(Map<String, String[]> parameterMap, BaseURL baseURL, String namespace,
		char separatorChar) {

		String faces23Namespace = namespace + separatorChar;

		Map<String, String[]> bridgeURLParameterMap = bridgeURI.getParameterMap();
		Set<Map.Entry<String, String[]>> parameterMapEntrySet = parameterMap.entrySet();

		for (Map.Entry<String, String[]> mapEntry : parameterMapEntrySet) {

			String parameterName = mapEntry.getKey();

			if (parameterName.startsWith(faces23Namespace)) {
				parameterName = parameterName.substring(faces23Namespace.length());
			}
			else if (parameterName.startsWith(namespace)) {
				parameterName = parameterName.substring(namespace.length());
			}

			// Note that preserved action parameters, parameters that already exist in the URL string,
			// and "javax.faces.ViewState" must not be copied.
			if (!ResponseStateManager.VIEW_STATE_PARAM.equals(parameterName) &&
					!bridgeURLParameterMap.containsKey(parameterName)) {
				baseURL.setParameter(parameterName, mapEntry.getValue());
			}
		}
	}

	private void copyURIParametersToBaseURL(List<URIParameter> uriParameters, BaseURL baseURL)
		throws MalformedURLException {

		for (URIParameter uriParameter : uriParameters) {
			String name = uriParameter.getName();
			String[] values = uriParameter.getValues();
			baseURL.setParameter(name, values);
			logger.debug("Copied parameter to baseURL name=[{0}] value=[{1}]", name, values);
		}
	}

	private PortletURL createActionURL(FacesContext facesContext, List<URIParameter> uriParameters)
		throws MalformedURLException {

		try {
			logger.debug("createActionURL uriParameters=[{0}]", uriParameters);

			ExternalContext externalContext = facesContext.getExternalContext();
			MimeResponse mimeResponse = (MimeResponse) externalContext.getResponse();
			PortletURL actionURL = mimeResponse.createActionURL();

			if (uriParameters != null) {
				copyURIParametersToBaseURL(uriParameters, actionURL);
			}

			// FACES-2978: Support render-redirect.
			for (URIParameter uriParameter : uriParameters) {

				if (viewIdRenderParameterName.equalsIgnoreCase(uriParameter.getName())) {
					String[] parameterValues = uriParameter.getValues();

					if ((parameterValues != null) && (parameterValues.length > 0)) {

						// TCK TestPage 179: redirectRenderPRP1Test
						return new FacesViewActionURL(actionURL, parameterValues[0]);
					}
				}
			}

			return actionURL;
		}
		catch (ClassCastException e) {
			throw new MalformedURLException(e.getMessage());
		}
	}

	private PortletURL createRenderURL(FacesContext facesContext, List<URIParameter> uriParameters)
		throws MalformedURLException {

		Bridge.PortletPhase portletRequestPhase = BridgeUtil.getPortletRequestPhase(facesContext);

		if (isHeaderOrRenderOrResourcePhase(portletRequestPhase)) {

			try {
				logger.debug("createRenderURL uriParameters=[{0}]", uriParameters);

				ExternalContext externalContext = facesContext.getExternalContext();
				MimeResponse mimeResponse = (MimeResponse) externalContext.getResponse();
				PortletURL renderURL = mimeResponse.createRenderURL();

				if (uriParameters != null) {
					copyURIParametersToBaseURL(uriParameters, renderURL);
				}

				return renderURL;
			}
			catch (ClassCastException e) {
				throw new MalformedURLException(e.getMessage());
			}
		}
		else {
			throw new MalformedURLException("Unable to create a RenderURL during " + portletRequestPhase.toString());
		}
	}

	private ResourceURL createResourceURL(FacesContext facesContext, List<URIParameter> uriParameters)
		throws MalformedURLException {

		try {
			logger.debug("createResourceURL uriParameters=[{0}]", uriParameters);

			// Ask the portlet container to create a portlet resource URL.
			ExternalContext externalContext = facesContext.getExternalContext();
			MimeResponse mimeResponse = (MimeResponse) externalContext.getResponse();
			ResourceURL resourceURL = mimeResponse.createResourceURL();

			// If the "javax.faces.resource" token is found in the URL, then
			String bridgeURIAsString = bridgeURI.toString();
			int tokenPos = bridgeURIAsString.indexOf("javax.faces.resource");

			if (tokenPos >= 0) {

				// Parse-out the resourceId
				String resourceId = bridgeURIAsString.substring(tokenPos);

				// Parse-out the resourceName and convert it to a URL parameter on the portlet resource URL.
				int queryStringPos = resourceId.indexOf('?');

				String resourceName = resourceId;

				if (queryStringPos > 0) {
					resourceName = resourceName.substring(0, queryStringPos);
				}

				int slashPos = resourceName.indexOf('/');

				if (slashPos > 0) {
					resourceName = resourceName.substring(slashPos + 1);
				}
				else {
					logger.debug("There is no slash after the [{0}] token in resourceURL=[{1}]", "javax.faces.resource",
						uriParameters);
				}

				resourceURL.setParameter("javax.faces.resource", resourceName);
				logger.debug("Added parameter to portletURL name=[{0}] value=[{1}]", "javax.faces.resource",
					resourceName);
			}

			// Copy the request parameters to the portlet resource URL.
			if (uriParameters != null) {
				copyURIParametersToBaseURL(uriParameters, resourceURL);
			}

			return resourceURL;
		}
		catch (ClassCastException e) {
			throw new MalformedURLException(e.getMessage());
		}
	}

	private List<URIParameter> getToStringParameters(boolean modeChanged) {
		return getToStringParameters(modeChanged, null);
	}

	private List<URIParameter> getToStringParameters(boolean modeChanged, Set<String> excludedParameterNames) {

		List<URIParameter> toStringParameters = new ArrayList<URIParameter>();
		Map<String, String[]> parameterMap = bridgeURI.getParameterMap();
		Set<Map.Entry<String, String[]>> entrySet = parameterMap.entrySet();
		boolean foundFacesViewIdParam = false;
		boolean foundFacesViewPathParam = false;

		for (Map.Entry<String, String[]> mapEntry : entrySet) {

			boolean addParameter;
			String parameterName = mapEntry.getKey();
			String[] parameterValues = mapEntry.getValue();
			String firstParameterValue = null;

			if ((parameterValues != null) && (parameterValues.length > 0)) {
				firstParameterValue = parameterValues[0];
			}

			if (Bridge.PORTLET_MODE_PARAMETER.equals(parameterName) ||
					Bridge.PORTLET_WINDOWSTATE_PARAMETER.equals(parameterName)) {
				addParameter = (firstParameterValue != null);
			}
			else if (Bridge.PORTLET_SECURE_PARAMETER.equals(parameterName)) {
				addParameter = ((firstParameterValue != null) &&
						("true".equalsIgnoreCase(firstParameterValue) ||
							"false".equalsIgnoreCase(firstParameterValue)));
			}
			else {

				if (!foundFacesViewIdParam) {
					foundFacesViewIdParam = Bridge.FACES_VIEW_ID_PARAMETER.equals(parameterName);
				}

				if (!foundFacesViewPathParam) {
					foundFacesViewPathParam = Bridge.FACES_VIEW_PATH_PARAMETER.equals(parameterName);
				}

				addParameter = true;
			}

			if ((addParameter) &&
					((excludedParameterNames == null) || !excludedParameterNames.contains(parameterName))) {
				toStringParameters.add(new URIParameter(parameterName, parameterValues));
			}
		}

		// If the "_jsfBridgeViewId" and "_jsfBridgeViewPath" parameters are not present in the URL, then add a
		// parameter that indicates the target Faces viewId.
		String viewId = getViewId();

		if (!foundFacesViewIdParam && !foundFacesViewPathParam && (viewId != null)) {

			// If the URI starts with the "portlet:" prefix, then
			if (bridgeURI.isPortletScheme()) {

				// TestPage220 (requestRenderRedisplayTest) - bridge-tck-scope-portlet
				//
				// If the bridge request scope goes from ACTION_PHASE -> RENDER_PHASE (rather than ACTION_PHASE ->
				// ACTION_PHASE), then the current view id needs to be added as a parameter on the URL.
				//
				// Detailed Explanation:
				//
				// The JSR 329 behavior for the bridge request scope was ACTION_PHASE -> ACTION_PHASE. That means that
				// re-renders of the portlet (for example, those invoked via RenderURL) in between action phases are
				// able to determine current view id (without explicitly adding it to the URL) because the bridge
				// request scope is responsible for restoring the last known UIViewRoot in the RENDER_PHASE.
				//
				// The JSR 378 behavior for the bridge request scope is ACTION_PHASE -> RENDER_PHASE. Since the bridge
				// request scope goes away at the end of the RENDER_PHASE, there is no way for RenderURL re-renders
				// to know the current view id without specifying it as a parameter on the RenderURL itself.
				boolean bridgeRequestScopeActionEnabled = PortletConfigParam.BridgeRequestScopeActionEnabled
					.getBooleanValue(portletConfig);

				if (!bridgeRequestScopeActionEnabled) {
					toStringParameters.add(new URIParameter(getViewIdParameterName(), viewId));
				}
			}

			// Otherwise,
			else {

				// Note that if the "javax.portlet.faces.PortletMode" parameter is specified, then a mode change is
				// being requested and the target Faces viewId parameter must NOT be added.
				if (!modeChanged) {

					String contextRelativePath = bridgeURI.getContextRelativePath(contextPath);

					if (contextRelativePath != null) {
						toStringParameters.add(new URIParameter(getViewIdParameterName(), contextRelativePath));
					}
				}
			}
		}

		return toStringParameters;
	}

	private boolean isViewPathMappedToFacesServlet(String viewPath) {

		// Try to determine the viewId by examining the servlet-mapping entries for the Faces Servlet.
		// For each servlet-mapping:
		for (ConfiguredServletMapping configuredFacesServletMapping : configuredFacesServletMappings) {

			// If the current servlet-mapping matches the viewPath, then
			logger.debug("Attempting to determine the facesViewId from {0}=[{1}]", Bridge.VIEW_PATH, viewPath);

			if (configuredFacesServletMapping.isMatch(viewPath)) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Determines whether or not the specified files have the same path (prefix) and extension (suffix).
	 *
	 * @param   file1  The first file to compare.
	 * @param   file2  The second file to compare.
	 *
	 * @return  <code>true</code> if the specified files have the same path (prefix) and extension (suffix), otherwise
	 *          <code>false</code>.
	 */
	private boolean matchPathAndExtension(String file1, String file2) {

		boolean match = false;

		String path1 = null;
		int lastSlashPos = file1.lastIndexOf("/");

		if (lastSlashPos > 0) {
			path1 = file1.substring(0, lastSlashPos);
		}

		String path2 = null;
		lastSlashPos = file2.lastIndexOf("/");

		if (lastSlashPos > 0) {
			path2 = file2.substring(0, lastSlashPos);
		}

		if (((path1 == null) && (path2 == null)) || ((path1 != null) && (path2 != null) && path1.equals(path2))) {

			String ext1 = null;
			int lastDotPos = file1.indexOf(".");

			if (lastDotPos > 0) {
				ext1 = file1.substring(lastDotPos);
			}

			String ext2 = null;
			lastDotPos = file2.indexOf(".");

			if (lastDotPos > 0) {
				ext2 = file2.substring(lastDotPos);
			}

			if (((ext1 == null) && (ext2 == null)) || ((ext1 != null) && (ext2 != null) && ext1.equals(ext2))) {
				match = true;
			}
		}

		return match;
	}

	private List<URIParameter> parameterMapToList(Map<String, String[]> parameterMap) {

		List<URIParameter> uriParameters = new ArrayList<URIParameter>();
		Set<Map.Entry<String, String[]>> entrySet = parameterMap.entrySet();

		for (Map.Entry<String, String[]> mapEntry : entrySet) {
			uriParameters.add(new URIParameter(mapEntry.getKey(), mapEntry.getValue()));
		}

		return uriParameters;
	}

	private static final class FacesViewActionURL extends PortletURLWrapper {

		// Private Final Data Members
		private final String viewId;

		public FacesViewActionURL(PortletURL portletURL, String viewId) {
			super(portletURL);
			this.viewId = viewId;
		}

		public String getViewId() {
			return viewId;
		}
	}

	private static final class URIParameter {

		// Private Final Data Members
		private final String name;
		private final String[] values;

		public URIParameter(String name, String value) {
			this(name, new String[] { value });
		}

		public URIParameter(String name, String[] values) {
			this.name = name;
			this.values = values;
		}

		public String getName() {
			return name;
		}

		public String[] getValues() {
			return values;
		}
	}
}
