/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.content.targeting.util;

import com.liferay.asset.kernel.exception.NoSuchVocabularyException;
import com.liferay.asset.kernel.model.AssetVocabulary;
import com.liferay.asset.kernel.service.AssetVocabularyLocalService;
import com.liferay.content.targeting.model.UserSegment;
import com.liferay.content.targeting.service.UserSegmentLocalServiceUtil;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.language.LanguageUtil;
import com.liferay.portal.kernel.model.Group;
import com.liferay.portal.kernel.search.Document;
import com.liferay.portal.kernel.search.Field;
import com.liferay.portal.kernel.search.Hits;
import com.liferay.portal.kernel.search.Indexer;
import com.liferay.portal.kernel.search.IndexerRegistryUtil;
import com.liferay.portal.kernel.service.ServiceContext;
import com.liferay.portal.kernel.service.ServiceContextThreadLocal;
import com.liferay.portal.kernel.util.ArrayUtil;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.LocaleUtil;
import com.liferay.portal.kernel.util.ResourceBundleLoader;
import com.liferay.portal.kernel.util.ResourceBundleUtil;
import com.liferay.portal.kernel.util.StringPool;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Eudaldo Alonso
 */
@Component(immediate = true, service = UserSegmentUtil.class)
public class UserSegmentUtil {

	public AssetVocabulary addAssetVocabulary(
			long userId, ServiceContext serviceContext)
		throws PortalException {

		serviceContext.setAddGroupPermissions(true);
		serviceContext.setAddGuestPermissions(true);

		AssetVocabulary assetVocabulary =
			_assetVocabularyLocalService.addVocabulary(
				userId, serviceContext.getScopeGroupId(), StringPool.BLANK,
				_getAssetVocabularyTitleMap(), null, StringPool.BLANK,
				serviceContext);

		return assetVocabulary;
	}

	public AssetVocabulary getAssetVocabulary(long groupId)
		throws PortalException {

		AssetVocabulary assetVocabulary = _getAssetVocabulary(
			groupId, UserSegmentConstants.VOCABULARY_NAME);

		if (assetVocabulary != null) {
			return assetVocabulary;
		}

		// Name has been replaced with site default title. See WCM-1385

		Map<Locale, String> titleMap = _getAssetVocabularyTitleMap();

		assetVocabulary = _getAssetVocabulary(
			groupId, titleMap.get(LocaleUtil.getSiteDefault()));

		if (assetVocabulary != null) {
			return assetVocabulary;
		}

		// Default locale has changed. See WCM-1313

		for (Map.Entry<Locale, String> entry : titleMap.entrySet()) {
			assetVocabulary = _getAssetVocabulary(groupId, entry.getValue());

			if (assetVocabulary != null) {
				return assetVocabulary;
			}
		}

		return assetVocabulary;
	}

	public long getAssetVocabularyId(long userId, ServiceContext serviceContext)
		throws PortalException {

		AssetVocabulary assetVocabulary = getAssetVocabulary(
			serviceContext.getScopeGroupId());

		if (assetVocabulary != null) {
			return assetVocabulary.getVocabularyId();
		}

		Group scopeGroup = serviceContext.getScopeGroup();

		String originalUuid = serviceContext.getUuid();

		if (scopeGroup.isStagingGroup()) {
			AssetVocabulary liveAssetVocabulary = getAssetVocabulary(
				scopeGroup.getLiveGroupId());

			if (liveAssetVocabulary == null) {
				ServiceContext serviceContextLive =
					(ServiceContext)serviceContext.clone();

				serviceContextLive.setScopeGroupId(scopeGroup.getLiveGroupId());

				liveAssetVocabulary = addAssetVocabulary(
					userId, serviceContextLive);
			}

			serviceContext.setUuid(liveAssetVocabulary.getUuid());
		}

		assetVocabulary = addAssetVocabulary(userId, serviceContext);

		serviceContext.setUuid(originalUuid);

		return assetVocabulary.getVocabularyId();
	}

	public long[] getAssetVocabularyIds(long[] groupIds)
		throws PortalException {

		List<Long> vocabularyIds = new ArrayList<>();

		for (long groupId : groupIds) {
			AssetVocabulary assetVocabulary = getAssetVocabulary(groupId);

			if (assetVocabulary != null) {
				vocabularyIds.add(assetVocabulary.getVocabularyId());
			}
		}

		if (vocabularyIds.isEmpty()) {
			ServiceContext serviceContext =
				ServiceContextThreadLocal.getServiceContext();

			if (serviceContext != null) {
				AssetVocabulary assetVocabulary = addAssetVocabulary(
					serviceContext.getUserId(), serviceContext);

				vocabularyIds.add(assetVocabulary.getVocabularyId());
			}
		}

		return ArrayUtil.toLongArray(vocabularyIds);
	}

	public List<UserSegment> getUserSegments(Hits hits) throws PortalException {
		List<Document> documents = hits.toList();

		List<UserSegment> userSegments = new ArrayList<>(documents.size());

		for (Document document : documents) {
			long userSegmentId = GetterUtil.getLong(
				document.get("userSegmentId"));

			UserSegment userSegment =
				UserSegmentLocalServiceUtil.fetchUserSegment(userSegmentId);

			if (userSegment == null) {
				userSegments = null;

				Indexer indexer = IndexerRegistryUtil.getIndexer(
					UserSegment.class);

				long companyId = GetterUtil.getLong(
					document.get(Field.COMPANY_ID));

				indexer.delete(companyId, document.getUID());
			}
			else if (userSegments != null) {
				userSegments.add(userSegment);
			}
		}

		return userSegments;
	}

	@Reference(unbind = "-")
	public void setAssetVocabularyService(
		AssetVocabularyLocalService assetVocabularyLocalService) {

		_assetVocabularyLocalService = assetVocabularyLocalService;
	}

	@Reference(
		target = "(bundle.symbolic.name=com.liferay.content.targeting.lang)",
		unbind = "-"
	)
	public void setResourceBundleLoader(
		ResourceBundleLoader resourceBundleLoader) {

		_resourceBundleLoader = resourceBundleLoader;
	}

	private AssetVocabulary _getAssetVocabulary(long groupId, String name)
		throws PortalException {

		// As of LPS-68908 this method is exposed in service

		try {
			return _assetVocabularyLocalService.getGroupVocabulary(
				groupId, name);
		}
		catch (NoSuchVocabularyException nsve) {
			return null;
		}
	}

	private Map<Locale, String> _getAssetVocabularyTitleMap() {
		Map<Locale, String> titleMap = new HashMap<>();

		Set<Locale> locales = LanguageUtil.getAvailableLocales();

		for (Locale locale : locales) {
			ResourceBundle resourceBundle =
				_resourceBundleLoader.loadResourceBundle(
					LanguageUtil.getLanguageId(locale));

			titleMap.put(
				locale,
				ResourceBundleUtil.getString(
					resourceBundle, UserSegmentConstants.VOCABULARY_TITLE));
		}

		return titleMap;
	}

	private AssetVocabularyLocalService _assetVocabularyLocalService;
	private ResourceBundleLoader _resourceBundleLoader;

}