/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.content.targeting.util;

import com.liferay.asset.kernel.AssetRendererFactoryRegistryUtil;
import com.liferay.asset.kernel.model.AssetCategory;
import com.liferay.asset.kernel.model.AssetRendererFactory;
import com.liferay.asset.kernel.model.ClassType;
import com.liferay.asset.kernel.model.ClassTypeReader;
import com.liferay.asset.kernel.service.AssetCategoryLocalServiceUtil;
import com.liferay.asset.kernel.util.comparator.AssetRendererFactoryTypeNameComparator;
import com.liferay.content.targeting.model.UserSegment;
import com.liferay.content.targeting.service.UserSegmentLocalServiceUtil;
import com.liferay.exportimport.kernel.staging.StagingUtil;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.language.LanguageUtil;
import com.liferay.portal.kernel.model.Group;
import com.liferay.portal.kernel.model.PortletConstants;
import com.liferay.portal.kernel.portlet.LiferayWindowState;
import com.liferay.portal.kernel.portlet.PortletProvider;
import com.liferay.portal.kernel.portlet.PortletProviderUtil;
import com.liferay.portal.kernel.security.permission.ResourceActionsUtil;
import com.liferay.portal.kernel.service.GroupLocalServiceUtil;
import com.liferay.portal.kernel.theme.ThemeDisplay;
import com.liferay.portal.kernel.util.ArrayUtil;
import com.liferay.portal.kernel.util.CharPool;
import com.liferay.portal.kernel.util.FriendlyURLNormalizerUtil;
import com.liferay.portal.kernel.util.ListUtil;
import com.liferay.portal.kernel.util.PortalUtil;
import com.liferay.portal.kernel.util.PredicateFilter;
import com.liferay.portal.kernel.util.ResourceBundleUtil;
import com.liferay.portal.kernel.util.StringBundler;
import com.liferay.portal.kernel.util.StringPool;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.portal.kernel.util.Validator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import javax.portlet.PortletMode;
import javax.portlet.PortletURL;

import javax.servlet.http.HttpServletRequest;

/**
 * @author Eudaldo Alonso
 */
public class ContentTargetingUtil {

	public static final String GUID_REPLACEMENT = "{ct_field_guid}";

	/**
	 * Use legacy code to escape attributes. Latest code from HtmlImpl breaks
	 * form builder.
	 *
	 * @see com.liferay.portal.util.HtmlImpl
	 * @param text
	 * @return
	 */
	public static String escapeAttribute(String text) {
		if (text == null) {
			return null;
		}

		if (text.length() == 0) {
			return StringPool.BLANK;
		}

		String prefix = "&#x";
		String postfix = StringPool.SEMICOLON;

		StringBuilder sb = new StringBuilder(text.length());

		for (int i = 0; i < text.length(); i++) {
			char c = text.charAt(i);

			if ((c > 255) || (c == CharPool.DASH) ||
				(c == CharPool.UNDERLINE) || Character.isLetterOrDigit(c)) {

				sb.append(c);
			}
			else {
				sb.append(prefix);

				String hexString = StringUtil.toHexString(c);

				if (hexString.length() == 1) {
					sb.append(StringPool.ASCII_TABLE[48]);
				}

				sb.append(hexString);
				sb.append(postfix);
			}
		}

		if (sb.length() == text.length()) {
			return text;
		}

		return sb.toString();
	}

	public static long[] getAncestorsAndCurrentGroupIds(long groupId)
		throws PortalException {

		Group scopeGroup = GroupLocalServiceUtil.fetchGroup(groupId);

		if (scopeGroup == null) {
			return null;
		}

		String stagedPortletId = StagingUtil.getStagedPortletId(
			PortletConstants.getRootPortletId(PortletKeys.CT_ADMIN));

		if (scopeGroup.isStagingGroup() &&
			!scopeGroup.isStagedPortlet(stagedPortletId)) {

			scopeGroup = scopeGroup.getLiveGroup();
		}

		List<Group> groups = new ArrayList<>();

		groups.addAll(scopeGroup.getAncestors());

		groups.add(scopeGroup);

		Group companyGroup = GroupLocalServiceUtil.getCompanyGroup(
			scopeGroup.getCompanyId());

		groups.add(companyGroup);

		long[] groupIds = new long[groups.size()];

		for (int i = 0; i < groups.size(); i++) {
			Group group = groups.get(i);

			groupIds[i] = group.getGroupId();
		}

		return groupIds;
	}

	public static long[] getAssetCategoryIds(List<UserSegment> userSegments) {
		long[] assetCategoryIds = new long[userSegments.size()];

		for (int i = 0; i < assetCategoryIds.length; i++) {
			UserSegment userSegment = userSegments.get(i);

			assetCategoryIds[i] = userSegment.getAssetCategoryId();
		}

		return assetCategoryIds;
	}

	public static long[] getAssetCategoryIds(
		long groupId, long[] userSegmentIds) {

		if (userSegmentIds == null) {
			return new long[0];
		}

		long[] assetCategoryIds = new long[userSegmentIds.length];

		for (int i = 0; i < userSegmentIds.length; i++) {
			UserSegment userSegment =
				UserSegmentLocalServiceUtil.fetchUserSegment(userSegmentIds[i]);

			if (userSegment != null) {
				assetCategoryIds[i] = userSegment.getAssetCategoryId(groupId);
			}
		}

		return assetCategoryIds;
	}

	public static String getAssetCategoryNames(
		long[] assetCategoryIds, Locale locale) {

		return getAssetCategoryNames(
			assetCategoryIds, locale, _CATEGORY_SEPARATOR);
	}

	public static String getAssetCategoryNames(
		long[] assetCategoryIds, Locale locale, String separator) {

		if (ArrayUtil.isEmpty(assetCategoryIds)) {
			return StringPool.BLANK;
		}

		StringBundler sb = new StringBundler((assetCategoryIds.length * 2) - 1);

		for (int i = 0; i < assetCategoryIds.length; i++) {
			AssetCategory assetCategory =
				AssetCategoryLocalServiceUtil.fetchAssetCategory(
					assetCategoryIds[i]);

			if (assetCategory == null) {
				continue;
			}

			sb.append(assetCategory.getTitle(locale));

			if (i != (assetCategoryIds.length - 1)) {
				sb.append(separator);
			}
		}

		return sb.toString();
	}

	public static List<AssetRendererFactory<?>> getAssetRendererFactories(
		HttpServletRequest request) {

		ThemeDisplay themeDisplay = (ThemeDisplay)request.getAttribute(
			WebKeys.THEME_DISPLAY);

		List<AssetRendererFactory<?>> assetRendererFactories =
			AssetRendererFactoryRegistryUtil.getAssetRendererFactories(
				themeDisplay.getCompanyId());

		assetRendererFactories = ListUtil.filter(
			assetRendererFactories,
			new PredicateFilter<AssetRendererFactory<?>>() {

				@Override
				public boolean filter(
					AssetRendererFactory<?> assetRendererFactory) {

					if (assetRendererFactory.isLinkable() &&
						assetRendererFactory.isSelectable()) {

						return true;
					}

					return false;
				}

			});

		return ListUtil.sort(
			assetRendererFactories,
			new AssetRendererFactoryTypeNameComparator(
				themeDisplay.getLocale()));
	}

	public static Map<String, Object> getAssetSelectorIconData(
			HttpServletRequest request,
			AssetRendererFactory assetRendererFactory, String index)
		throws Exception {

		return getAssetSelectorIconData(
			request, assetRendererFactory, index, false);
	}

	public static Map<String, Object> getAssetSelectorIconData(
			HttpServletRequest request,
			AssetRendererFactory assetRendererFactory, String index,
			boolean instantiable)
		throws Exception {

		ThemeDisplay themeDisplay = (ThemeDisplay)request.getAttribute(
			WebKeys.THEME_DISPLAY);

		PortletURL assetBrowserURL = getAssetBrowserURL(
			request, assetRendererFactory.getClassName(), instantiable);

		Map<String, Object> data = new HashMap<>();

		data.put("groupid", String.valueOf(themeDisplay.getScopeGroupId()));
		data.put("href", assetBrowserURL.toString());

		if (!Validator.isBlank(index)) {
			data.put("index", index);
		}

		String typeName = assetRendererFactory.getTypeName(
			themeDisplay.getLocale());

		data.put(
			"title",
			LanguageUtil.format(
				themeDisplay.getLocale(), "select-x", typeName, false));
		data.put("type", typeName);

		return data;
	}

	public static String getCategorySeparator() {
		return _CATEGORY_SEPARATOR;
	}

	public static String getDescription(Class<?> clazz, Locale locale) {
		String key = clazz.getName().concat(".description");

		String description = getModelResource(locale, clazz, key);

		if (description.endsWith(key)) {
			description = getShortDescription(clazz, locale);
		}

		return description;
	}

	public static String getModelResource(
		Locale locale, Class clazz, String name) {

		String key = ResourceActionsUtil.getModelResourceNamePrefix() + name;

		try {
			ResourceBundle resourceBundle = ResourceBundleUtil.getBundle(
				"content.Language", locale, clazz);

			String modelResource = ResourceBundleUtil.getString(
				resourceBundle, key);

			if (modelResource == null) {
				modelResource = key;
			}

			return modelResource;
		}
		catch (MissingResourceException mre) {
			return ResourceActionsUtil.getModelResource(locale, name);
		}
	}

	public static String getName(Class<?> clazz, Locale locale) {
		return getModelResource(locale, clazz, clazz.getName());
	}

	public static List<Map<String, Object>> getSelectorEntries(
			HttpServletRequest request, ResourceBundle resourceBundle,
			String index, boolean instantiable)
		throws Exception {

		ThemeDisplay themeDisplay = (ThemeDisplay)request.getAttribute(
			WebKeys.THEME_DISPLAY);

		List<Map<String, Object>> selectorEntries = new ArrayList<>();

		for (AssetRendererFactory<?> assetRendererFactory :
				getAssetRendererFactories(request)) {

			if (assetRendererFactory.isSupportsClassTypes()) {
				selectorEntries.addAll(
					_getSelectorEntries(
						assetRendererFactory, request, resourceBundle,
						themeDisplay, index, instantiable));
			}
			else {
				Map<String, Object> selectorEntry = new HashMap<>();

				selectorEntry.put(
					"data",
					_getSelectorEntryData(
						assetRendererFactory, request, resourceBundle,
						themeDisplay, index, instantiable));
				selectorEntry.put(
					"iconCssClass",
					_getSelectorEntryIconCssClass(assetRendererFactory));
				selectorEntry.put(
					"id",
					_getSelectorEntryId(
						assetRendererFactory, themeDisplay, index));
				selectorEntry.put(
					"message",
					_getSelectorEntryMessage(
						assetRendererFactory, themeDisplay));

				selectorEntries.add(selectorEntry);
			}
		}

		return selectorEntries;
	}

	public static String getShortDescription(Class<?> clazz, Locale locale) {
		String key = clazz.getName().concat(".shortDescription");

		String shortDescription = getModelResource(locale, clazz, key);

		if (shortDescription.endsWith(key)) {
			shortDescription = StringPool.BLANK;
		}

		return shortDescription;
	}

	protected static PortletURL getAssetBrowserURL(
			HttpServletRequest request, String className, boolean instantiable)
		throws Exception {

		ThemeDisplay themeDisplay = (ThemeDisplay)request.getAttribute(
			WebKeys.THEME_DISPLAY);

		PortletURL assetBrowserURL = PortletProviderUtil.getPortletURL(
			request, className, PortletProvider.Action.BROWSE);

		assetBrowserURL.setParameter(
			"groupId", String.valueOf(themeDisplay.getScopeGroupId()));
		assetBrowserURL.setParameter(
			"selectedGroupIds",
			StringUtil.merge(
				PortalUtil.getSharedContentSiteGroupIds(
					themeDisplay.getCompanyId(), themeDisplay.getScopeGroupId(),
					themeDisplay.getUserId())));
		assetBrowserURL.setParameter(
			"showNonindexable", String.valueOf(Boolean.TRUE));

		String eventName = "selectContent";

		if (instantiable) {
			eventName =
				StringPool.UNDERLINE + GUID_REPLACEMENT + StringPool.UNDERLINE +
					eventName;
		}

		assetBrowserURL.setParameter("eventName", eventName);
		assetBrowserURL.setParameter("typeSelection", className);
		assetBrowserURL.setPortletMode(PortletMode.VIEW);
		assetBrowserURL.setWindowState(LiferayWindowState.POP_UP);

		return assetBrowserURL;
	}

	private static long _getAssetBrowserGroupId(
		AssetRendererFactory<?> assetRendererFactory,
		ThemeDisplay themeDisplay) {

		Group scopeGroup = themeDisplay.getScopeGroup();

		boolean stagedLocally = false;
		boolean stagedReferrerPortlet = false;

		if (scopeGroup.isStaged() && !scopeGroup.isStagedRemotely()) {
			stagedLocally = true;

			stagedReferrerPortlet = scopeGroup.isStagedPortlet(
				assetRendererFactory.getPortletId());
		}

		long groupId = scopeGroup.getGroupId();

		if (stagedLocally && scopeGroup.isStagingGroup()) {
			boolean stagedReferencePortlet = scopeGroup.isStagedPortlet(
				assetRendererFactory.getPortletId());

			if (stagedReferrerPortlet && !stagedReferencePortlet) {
				groupId = scopeGroup.getLiveGroupId();
			}
		}

		return groupId;
	}

	private static PortletURL _getAssetBrowserPortletURL(
			AssetRendererFactory<?> assetRendererFactory,
			HttpServletRequest request, ThemeDisplay themeDisplay,
			boolean instantiable)
		throws Exception {

		PortletURL portletURL = PortletProviderUtil.getPortletURL(
			request, assetRendererFactory.getClassName(),
			PortletProvider.Action.BROWSE);

		if (portletURL == null) {
			return portletURL;
		}

		long groupId = _getAssetBrowserGroupId(
			assetRendererFactory, themeDisplay);

		String eventName = "selectContent";

		if (instantiable) {
			eventName =
				StringPool.UNDERLINE + ContentTargetingUtil.GUID_REPLACEMENT +
					StringPool.UNDERLINE + "selectContent";
		}

		portletURL.setParameter("eventName", eventName);
		portletURL.setParameter("groupId", String.valueOf(groupId));
		portletURL.setParameter(
			"selectedGroupIds",
			StringUtil.merge(
				PortalUtil.getSharedContentSiteGroupIds(
					themeDisplay.getCompanyId(), groupId,
					themeDisplay.getUserId())));
		portletURL.setParameter(
			"showNonindexable", String.valueOf(Boolean.TRUE));

		portletURL.setParameter(
			"typeSelection", assetRendererFactory.getClassName());
		portletURL.setPortletMode(PortletMode.VIEW);
		portletURL.setWindowState(LiferayWindowState.POP_UP);

		return portletURL;
	}

	private static List<Map<String, Object>> _getSelectorEntries(
			AssetRendererFactory<?> assetRendererFactory,
			HttpServletRequest request, ResourceBundle resourceBundle,
			ThemeDisplay themeDisplay, String index, boolean instantiable)
		throws Exception {

		long groupId = _getAssetBrowserGroupId(
			assetRendererFactory, themeDisplay);

		ClassTypeReader classTypeReader =
			assetRendererFactory.getClassTypeReader();

		List<ClassType> classTypes = classTypeReader.getAvailableClassTypes(
			PortalUtil.getCurrentAndAncestorSiteGroupIds(groupId),
			themeDisplay.getLocale());

		if (classTypes.isEmpty()) {
			return Collections.emptyList();
		}

		List<Map<String, Object>> selectorEntries = new ArrayList<>();

		for (ClassType classType : classTypes) {
			Map<String, Object> selectorEntry = new HashMap<>();

			selectorEntry.put(
				"data",
				_getSelectorEntryData(
					assetRendererFactory, request, resourceBundle, themeDisplay,
					classType, index, instantiable));
			selectorEntry.put(
				"iconCssClass",
				_getSelectorEntryIconCssClass(assetRendererFactory));
			selectorEntry.put("id", _getSelectorEntryId(classType, index));
			selectorEntry.put("message", _getSelectorEntryMessage(classType));

			selectorEntries.add(selectorEntry);
		}

		return selectorEntries;
	}

	private static Map<String, Object> _getSelectorEntryData(
			AssetRendererFactory<?> assetRendererFactory,
			HttpServletRequest request, ResourceBundle resourceBundle,
			ThemeDisplay themeDisplay, ClassType classType, String index,
			boolean instantiable)
		throws Exception {

		Map<String, Object> selectorEntryData = new HashMap<>();

		PortletURL portletURL = _getAssetBrowserPortletURL(
			assetRendererFactory, request, themeDisplay, instantiable);

		if (portletURL != null) {
			portletURL.setParameter(
				"subtypeSelectionId",
				String.valueOf(classType.getClassTypeId()));

			selectorEntryData.put("href", portletURL.toString());
		}

		selectorEntryData.put("constrain", true);
		selectorEntryData.put("destroyOnHide", true);
		selectorEntryData.put(
			"groupid", String.valueOf(themeDisplay.getScopeGroupId()));

		if (!Validator.isBlank(index)) {
			selectorEntryData.put("index", index);
		}

		selectorEntryData.put("modal", true);
		selectorEntryData.put(
			"title",
			LanguageUtil.format(
				resourceBundle, "select-x", classType.getName(), false));

		selectorEntryData.put("type", classType.getName());

		return selectorEntryData;
	}

	private static Map<String, Object> _getSelectorEntryData(
			AssetRendererFactory<?> assetRendererFactory,
			HttpServletRequest request, ResourceBundle resourceBundle,
			ThemeDisplay themeDisplay, String index, boolean instantiable)
		throws Exception {

		Map<String, Object> selectorEntryData = new HashMap<>();

		PortletURL assetBrowserPortletURL = _getAssetBrowserPortletURL(
			assetRendererFactory, request, themeDisplay, instantiable);

		if (assetBrowserPortletURL != null) {
			selectorEntryData.put("href", assetBrowserPortletURL.toString());
		}

		String typeName = assetRendererFactory.getTypeName(
			themeDisplay.getLocale());

		selectorEntryData.put("constrain", true);
		selectorEntryData.put("destroyOnHide", true);
		selectorEntryData.put(
			"groupid", String.valueOf(themeDisplay.getScopeGroupId()));

		if (!Validator.isBlank(index)) {
			selectorEntryData.put("index", index);
		}

		selectorEntryData.put("modal", true);
		selectorEntryData.put(
			"title",
			LanguageUtil.format(resourceBundle, "select-x", typeName, false));

		selectorEntryData.put(
			"type", assetRendererFactory.getTypeName(themeDisplay.getLocale()));

		return selectorEntryData;
	}

	private static String _getSelectorEntryIconCssClass(
		AssetRendererFactory<?> assetRendererFactory) {

		return assetRendererFactory.getIconCssClass();
	}

	private static String _getSelectorEntryId(
		AssetRendererFactory<?> assetRendererFactory, ThemeDisplay themeDisplay,
		String index) {

		String selectorEntryId = "groupId_";

		selectorEntryId += FriendlyURLNormalizerUtil.normalize(
			assetRendererFactory.getTypeName(themeDisplay.getLocale()));

		selectorEntryId += "_" + index;

		return selectorEntryId;
	}

	private static String _getSelectorEntryId(
		ClassType classType, String index) {

		String selectorEntryId = "groupId_";

		selectorEntryId += FriendlyURLNormalizerUtil.normalize(
			classType.getName());

		selectorEntryId += "_" + index;

		return selectorEntryId;
	}

	private static String _getSelectorEntryMessage(
		AssetRendererFactory<?> assetRendererFactory,
		ThemeDisplay themeDisplay) {

		return assetRendererFactory.getTypeName(themeDisplay.getLocale());
	}

	private static String _getSelectorEntryMessage(ClassType classType) {
		return classType.getName();
	}

	private static final String _CATEGORY_SEPARATOR = "_CATEGORY_";

}