/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.user.associated.data.display;

import com.liferay.portal.kernel.dao.orm.DynamicQuery;
import com.liferay.portal.kernel.model.BaseModel;
import com.liferay.user.associated.data.util.UADDynamicQueryUtil;

import java.io.Serializable;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

/**
 * The base implementation of {@link UADDisplay} for entities generated with
 * {@code ServiceBuilder}. The count and retrieval are based on
 * {@link DynamicQuery}, which is available in the service generated by
 * {@code ServiceBuilder}.
 *
 * @author Pei-Jung Lan
 * @review
 */
public abstract class BaseModelUADDisplay<T extends BaseModel>
	implements UADDisplay<T> {

	@Override
	public long count(long userId) {
		return doCount(getDynamicQuery(userId));
	}

	@Override
	public Map<String, Object> getFieldValues(T t, String[] fieldNames) {
		Map<String, Object> modelAttributes = t.getModelAttributes();

		Set<String> modelAttributesKeySet = modelAttributes.keySet();

		modelAttributesKeySet.retainAll(Arrays.asList(fieldNames));

		return modelAttributes;
	}

	@Override
	public Serializable getPrimaryKey(T baseModel) {
		return baseModel.getPrimaryKeyObj();
	}

	@Override
	public List<T> getRange(long userId, int start, int end) {
		return doGetRange(getDynamicQuery(userId), start, end);
	}

	@Override
	public String getTypeName(Locale locale) {
		return getTypeClass().getSimpleName();
	}

	/**
	 * Returns the number of entities of type {@code T} associated with a user
	 * using a {@link DynamicQuery}.
	 *
	 * @param dynamicQuery the dynamicQuery to be passed to the service layer
	 * @return
	 * @review
	 */
	protected abstract long doCount(DynamicQuery dynamicQuery);

	/**
	 * Returns a new {@link DynamicQuery} from the relevant service for type
	 * {@code T}.
	 *
	 * @return a new {@link DynamicQuery} to be used by the service layer
	 * @review
	 */
	protected abstract DynamicQuery doGetDynamicQuery();

	/**
	 * Returns entities of type {@code T} in the given range associated with a
	 * user using a {@link DynamicQuery}.
	 *
	 * @param dynamicQuery the dynamicQuery to be passed to the service layer
	 * @param start the starting index of the result set, for pagination
	 * @param end the ending index of the result set, for pagination
	 * @return a paginated list of entities of entities of type {@code T}
	 * @review
	 */
	protected abstract List<T> doGetRange(
		DynamicQuery dynamicQuery, int start, int end);

	/**
	 * Returns names identifying fields on the entity of type {@code T}
	 * that contain the primary key of a user.
	 *
	 * @return fields that may contain the primary key of a user
	 * @review
	 */
	protected abstract String[] doGetUserIdFieldNames();

	/**
	 * Returns a {@link DynamicQuery} for type {@code T}. It should be populated
	 * with criterion and ready for use by the service.
	 *
	 * @param userId the the primary key of a user used to pre-filter the
	 * 				 {@link DynamicQuery}
	 * @return a pre-filtered {@link DynamicQuery}
	 * @review
	 */
	protected DynamicQuery getDynamicQuery(long userId) {
		return UADDynamicQueryUtil.addDynamicQueryCriteria(
			doGetDynamicQuery(), doGetUserIdFieldNames(), userId);
	}

}