package com.launchdarkly.sdk.server.interfaces;

import com.launchdarkly.sdk.EvaluationReason;
import com.launchdarkly.sdk.LDUser;
import com.launchdarkly.sdk.LDValue;

/**
 * Base class for all analytics events that are generated by the client. Also defines all of its own subclasses.
 * 
 * Applications do not need to reference these types directly. They are used internally in analytics event
 * processing, and are visible only to support writing a custom implementation of {@link EventProcessor} if
 * desired.
 */
public class Event {
  private final long creationDate;
  private final LDUser user;

  /**
   * Base event constructor.
   * @param creationDate the timestamp in milliseconds
   * @param user the user associated with the event
   */
  public Event(long creationDate, LDUser user) {
    this.creationDate = creationDate;
    this.user = user;
  }
  
  /**
   * The event timestamp.
   * @return the timestamp in milliseconds
   */
  public long getCreationDate() {
    return creationDate;
  }
  
  /**
   * The user associated with the event.
   * @return the user object
   */
  public LDUser getUser() {
    return user;
  }

  /**
   * Convert a user into a context kind string
   * @param user the user to get the context kind from
   * @return the context kind string
   */
  private static final String computeContextKind(LDUser user) {
    return user != null && user.isAnonymous() ? "anonymousUser" : "user";
  }
  
  /**
   * A custom event created with {@link LDClientInterface#track(String, LDUser)} or one of its overloads.
   */
  public static final class Custom extends Event {
    private final String key;
    private final LDValue data;
    private final Double metricValue;
    private final String contextKind;

    /**
     * Constructs a custom event.
     * @param timestamp the timestamp in milliseconds
     * @param key the event key
     * @param user the user associated with the event
     * @param data custom data if any (null is the same as {@link LDValue#ofNull()})
     * @param metricValue custom metric value if any
     * @since 4.8.0
     */
    public Custom(long timestamp, String key, LDUser user, LDValue data, Double metricValue) {
      super(timestamp, user);
      this.key = key;
      this.data = LDValue.normalize(data);
      this.metricValue = metricValue;
      this.contextKind = computeContextKind(user);
    }

    /**
     * The custom event key.
     * @return the event key
     */
    public String getKey() {
      return key;
    }
    
    /**
     * The custom data associated with the event, if any.
     * @return the event data (null is equivalent to {@link LDValue#ofNull()})
     */
    public LDValue getData() {
      return data;
    }
    
    /**
     * The numeric metric value associated with the event, if any.
     * @return the metric value or null
     */
    public Double getMetricValue() {
      return metricValue;
    }

    /**
     * The context kind of the user that generated this event
     * @return the context kind
     */
    public String getContextKind() {
      return contextKind;
    }
  }

  /**
   * An event created with {@link LDClientInterface#identify(LDUser)}.
   */
  public static final class Identify extends Event {
    /**
     * Constructs an identify event.
     * @param timestamp the timestamp in milliseconds
     * @param user the user associated with the event
     */
    public Identify(long timestamp, LDUser user) {
      super(timestamp, user);
    }
  }

  /**
   * An event created internally by the SDK to hold user data that may be referenced by multiple events.
   */
  public static final class Index extends Event {
    /**
     * Constructs an index event.
     * @param timestamp the timestamp in milliseconds
     * @param user the user associated with the event
     */
    public Index(long timestamp, LDUser user) {
      super(timestamp, user);
    }
  }
  
  /**
   * An event generated by a feature flag evaluation.
   */
  public static final class FeatureRequest extends Event {
    private final String key;
    private final int variation;
    private final LDValue value;
    private final LDValue defaultVal;
    private final int version;
    private final String prereqOf;
    private final boolean trackEvents;
    private final long debugEventsUntilDate;
    private final EvaluationReason reason;
    private final boolean debug;
    private final String contextKind;

    /**
     * Constructs a feature request event.
     * @param timestamp the timestamp in milliseconds
     * @param key the flag key
     * @param user the user associated with the event
     * @param version the flag version, or -1 if the flag was not found
     * @param variation the result variation, or -1 if there was an error
     * @param value the result value
     * @param defaultVal the default value passed by the application
     * @param reason the evaluation reason, if it is to be included in the event
     * @param prereqOf if this flag was evaluated as a prerequisite, this is the key of the flag that referenced it
     * @param trackEvents true if full event tracking is turned on for this flag
     * @param debugEventsUntilDate if non-null, the time until which event debugging should be enabled
     * @param debug true if this is a debugging event
     * @since 4.8.0
     */
    public FeatureRequest(long timestamp, String key, LDUser user, int version, int variation, LDValue value,
        LDValue defaultVal, EvaluationReason reason, String prereqOf, boolean trackEvents, long debugEventsUntilDate, boolean debug) {
      super(timestamp, user);
      this.key = key;
      this.version = version;
      this.variation = variation;
      this.value = value;
      this.defaultVal = defaultVal;
      this.prereqOf = prereqOf;
      this.trackEvents = trackEvents;
      this.debugEventsUntilDate = debugEventsUntilDate;
      this.reason = reason;
      this.debug = debug;
      this.contextKind = computeContextKind(user);
    }

    /**
     * The key of the feature flag that was evaluated.
     * @return the flag key
     */
    public String getKey() {
      return key;
    }

    /**
     * The index of the selected flag variation, or -1 if the application default value was used.
     * @return zero-based index of the variation, or -1
     */
    public int getVariation() {
      return variation;
    }

    /**
     * The value of the selected flag variation.
     * @return the value
     */
    public LDValue getValue() {
      return value;
    }

    /**
     * The application default value used in the evaluation.
     * @return the application default
     */
    public LDValue getDefaultVal() {
      return defaultVal;
    }

    /**
     * The version of the feature flag that was evaluated, or -1 if the flag was not found.
     * @return the flag version or null
     */
    public int getVersion() {
      return version;
    }

    /**
     * If this flag was evaluated as a prerequisite for another flag, the key of the other flag.
     * @return a flag key or null
     */
    public String getPrereqOf() {
      return prereqOf;
    }

    /**
     * True if full event tracking is enabled for this flag.
     * @return true if full event tracking is on
     */
    public boolean isTrackEvents() {
      return trackEvents;
    }

    /**
     * If debugging is enabled for this flag, the Unix millisecond time at which to stop debugging.
     * @return a timestamp or zero
     */
    public long getDebugEventsUntilDate() {
      return debugEventsUntilDate;
    }

    /**
     * The {@link EvaluationReason} for this evaluation, or null if the reason was not requested for this evaluation. 
     * @return a reason object or null
     */
    public EvaluationReason getReason() {
      return reason;
    }

    /**
     * True if this event was generated due to debugging being enabled.
     * @return true if this is a debug event
     */
    public boolean isDebug() {
      return debug;
    }

    /**
     * The context kind of the user that generated this event
     * @return the context kind
     */
    public String getContextKind() {
      return contextKind;
    }
  }

  /**
   * An event generated by aliasing users
   * @since 5.4.0
   */
  public static final class AliasEvent extends Event {
    private final String key;
    private final String contextKind;
    private final String previousKey;
    private final String previousContextKind;

    /**
     * Constructs an alias event.
     * @param timestamp when the event was created
     * @param user the user being aliased to
     * @param previousUser the user being aliased from
     */
    public AliasEvent(long timestamp, LDUser user, LDUser previousUser) {
      super(timestamp, user);
      this.key = user.getKey();
      this.contextKind = computeContextKind(user);
      this.previousKey = previousUser.getKey();
      this.previousContextKind = computeContextKind(previousUser);
    }

    /**
     * Get the key of the user being aliased to
     * @return the user key
     */
    public String getKey() {
      return key;
    }

    /**
     * Get the kind of the user being aliased to
     * @return the context kind
     */
    public String getContextKind() {
      return contextKind;
    }

    /**
     * Get the key of the user being aliased from
     * @return the previous user key
     */
    public String getPreviousKey() {
      return previousKey;
    }

    /**
     * Get the kind of the user being aliased from
     * @return the previous context kind
     */
    public String getPreviousContextKind() {
      return previousContextKind;
    }
  }
}