package com.kynetics.awscodeartifactsbtplugin

import sbt.Keys.*
import sbt.librarymanagement.MavenRepository
import sbt.util.CacheImplicits.*
import sbt.{AutoPlugin, Credentials, Def, Setting, SettingKey, Task, TaskKey, Tracked, settingKey, taskKey}
import software.amazon.awssdk.regions.Region
import software.amazon.awssdk.services.codeartifact.CodeartifactClient
import software.amazon.awssdk.services.codeartifact.model.{GetAuthorizationTokenRequest, PackageVersionStatus, UpdatePackageVersionsStatusRequest}

import java.time.{Duration, Instant}

object AwsCodeArtifactSbtPlugin extends AutoPlugin {
  override def trigger = allRequirements

  object autoImport {
    val awsRegion: SettingKey[String] =
      settingKey[String]("Region of the CodeArtifact service.")
    val awsOwner: SettingKey[String] =
      settingKey[String]("Owner of the AWS account.")
    val awsArtifactDomain: SettingKey[String] =
      settingKey[String]("Aws CodeArtifact domain.")
    val awsArtifactRepoName: SettingKey[String] =
      settingKey[String]("Aws CodeArtifact repository name.")
    val tokenDuration: SettingKey[Duration] = settingKey[Duration](
      "The CodeArtifact repository authorization token duration. It must be between 15 minutes and 12 hours"
    )
    val isSbtPlugin: SettingKey[Boolean] = settingKey[Boolean](
      "True if we're using the plug-in to publish another SBT plug-in, false otherwise."
    )
    val publishPlugin: TaskKey[Unit] = taskKey[Unit]("Publish the artifacts of this SBT plug-in.")
  }

  import autoImport.*

  private lazy val codeArtifactClient = Def.setting {
    CodeartifactClient.builder()
      .region(Region.of(awsRegion.value))
      .build()
  }

  private def authorizationToken: Def.Initialize[Task[String]] = Def.task[String] {
    streams.value.log.info("Provide eventually cached AWS Token")

    def request = GetAuthorizationTokenRequest.builder()
      .domain(awsArtifactDomain.value)
      .domainOwner(awsOwner.value)
      .durationSeconds(tokenDuration.value.toSeconds)
      .build()

    def get = codeArtifactClient.value.getAuthorizationToken(request).authorizationToken()
    def refresh = {
      streams.value.log.info("Refresh AWS Token")
      (get, Instant.now.plus(tokenDuration.value.minusMinutes(1)))
    }
    def expired(i: Instant) = Instant.now.isAfter(i)
    val store = streams.value.cacheStoreFactory.make("AwsCodeArtifactSbtPlugin")
    Tracked.lastOutput[Unit, (String, Instant)](store) {
      case (_, None)                           => refresh
      case (_, Some((_, exp))) if expired(exp) => refresh
      case (_, Some(oldValue))                 => oldValue
    }.andThen(_._1)(())
  }

  private def awsArtifactRepoDomainName: Def.Initialize[String] = Def.setting {
    val domain = awsArtifactDomain.value
    s"$domain-${awsOwner.value}.d.codeartifact.${awsRegion.value}.amazonaws.com"
  }

  private def mavenRepository: Def.Initialize[MavenRepository] = Def.setting {
    val domain = awsArtifactDomain.value
    val repoName = awsArtifactRepoName.value
    val repoUrl = awsArtifactRepoDomainName.value
    MavenRepository(
      s"$domain-$repoName",
      s"https://$repoUrl/maven/$repoName",
    )
  }

  @SuppressWarnings(Array("org.wartremover.warts.Equals"))
  private def publishConf = Def.task {
    val oldConf = publishConfiguration.value

    if (isSbtPlugin.value) {
      // If we're publishing a plug-in, let's remove the artifacts generated by SBT (with the wrong name) and use
      // those generated by sbt-vspp (where the Scala and SBT binary versions are appended to the name)
      oldConf.withArtifacts(
        // Discard the artifacts whose name is equal to the project name, as we want to keep those with the Scala/SBT version
        // Wrong name: aws-codeartifact-sbt-plugin
        // Right name: aws-codeartifact-sbt-plugin_2.12_1.0
        oldConf.artifacts.filterNot(_._1.name == name.value)
      )
    } else {
      // Return the old configuration if we're not pushing an SBT plug-in, as the name of the artifacts generated by
      // SBT are just fine for libraries
      oldConf
    }
  }

  @SuppressWarnings(Array(
    "org.wartremover.warts.Throw",
    "org.wartremover.warts.TraversableOps",
    "org.wartremover.warts.IterableOps"
  ))
  private def publishPluginImpl: Def.Initialize[Task[Unit]] =  Def.task {
    if (isSbtPlugin.value) { // run this task only if the project is a plug-in
      streams.value.log.info("Marking the new package version as Published")

      // As the artifacts were filtered to keep those generated by sbt-vspp, take the first one and extract its name
      // publishConf.value.artifacts returns a list of pairs (Artifact, File)
      val packageName = publishConf.value.artifacts.head._1.name

      val request = UpdatePackageVersionsStatusRequest.builder()
        .domain(awsArtifactDomain.value)
        .domainOwner(awsOwner.value)
        .repository(awsArtifactRepoName.value)
        .format("maven")
        .namespace("com.kynetics")
        .packageValue(packageName)
        .versions(version.value)
        .targetStatus(PackageVersionStatus.PUBLISHED)
        .build()

      val result = codeArtifactClient.value.updatePackageVersionsStatus(request)
      result.successfulVersions().forEach { case (version, versionInfo) =>
        streams.value.log.info(s"Correctly updated $version to ${versionInfo.status().toString}")
      }
    } else {
      streams.value.log.error("This project is not an SBT plug-in, aborting")
      throw new IllegalStateException("This project is not an SBT plug-in, aborting")
    }
  }.dependsOn(publish)

  override lazy val globalSettings: Seq[Setting[_]] = Seq(
    isSbtPlugin := false
  )

  override lazy val projectSettings : Seq[Setting[_]] = Seq(
    publishConfiguration := publishConf.value,
    publishTo := Some(mavenRepository.value),
    credentials += Credentials(
      s"${awsArtifactDomain.value}/${awsArtifactRepoName.value}",
      awsArtifactRepoDomainName.value,
      "aws",
      authorizationToken.value
    ),
    resolvers += mavenRepository.value,
    publishPlugin := publishPluginImpl.value
  )
}
