package com.jamonapi.utils;

import java.sql.*;
import java.util.*;
import com.jamonapi.*;

/**
 * Purpose:     Provides various methods for obtaining resultset data.
 * Author:      Andy Evans
 * Date:        3/28/2001
 */

public class ResultSetUtils extends java.lang.Object {
    static ResultSetUtils resultSetUtils = new ResultSetUtils();
    
    protected ResultSetUtils() {
    }
    
    public static ResultSetUtils createInstance() {
        return resultSetUtils;
    }
    
    /**
     * The following method returns an ArrayList containing the ResultSet column names and data.
     * The first row of the ArrayList will always contain a string array of the column names.
     * Subsequent rows contain ResultSet rows as object arrays. The getColumnNames() method is
     * called to get the column names string array, and the getRowData() to convert ResultSet
     * row data into an object array.
     */
    public void fullResultSetToArrayList(ArrayList arrayList, ResultSet resultSet) throws SQLException {
        
        resultSetMetaDataToArrayList(arrayList, resultSet);
        resultSetToArrayList(arrayList, resultSet);
        
    }
    
    
    private Monitor start(String locator) {
        return AppConstants.start("ResultSetUtils."+locator);
    }
    
    
    
    /**
     * The following method returns an ArrayList containing the ResultSet column names and data.
     * The first row of the ArrayList will always contain a string array of the column names.
     * Subsequent rows contain ResultSet rows as object arrays. The getColumnNames() method is
     * called to get the column names string array, and the getRowData() to convert ResultSet
     * row data into an object array.
     */
    public void resultSetToArrayList(ArrayList arrayList, ResultSet resultSet) throws SQLException {
        //Monitor mon=start("resultSetToArrayList");
        ResultSetMetaData resultSetMetaData = resultSet.getMetaData();
        int columnCount = resultSetMetaData.getColumnCount();
        
        //Loop through the ResultSet data rows and add each one to the ArrayList
        while(resultSet.next()) {
            arrayList.add(getRowData(resultSet, columnCount));
        }
        
        //mon.stop();
        
        
    }
    
    
    /**
     * The following method returns an ArrayList containing the ResultSetMetaData column names.
     * There is only 1 row in the ArrayList and it will always contain a string array of the column names.
     */
    public void resultSetMetaDataToArrayList(ArrayList arrayList, ResultSet resultSet) throws SQLException {
        ResultSetMetaData resultSetMetaData = resultSet.getMetaData();
        int columnCount = resultSetMetaData.getColumnCount();
        
        //Add the String array of column names to the ArrayList
        arrayList.add(getColumnNames(resultSetMetaData));
        
    }
    
    
    /**
     * The following method converts ResultSet data to a Map of object keys and values using an instance
     * of HashMap.
     *
     * The first column in the ResultSet supplies the key, the second column the value. If the ResultSet
     * contains more than two columns then subsequent columns are ignored. If the ResultSet supplies
     * only one column then an SQLException is thrown. The ResultSet values are converted to objects
     * using the getRowData() method which converts a ResultSet row into an array of objects.
     */
    public void resultSetToMap(Map map, ResultSet resultSet) throws SQLException {
        int columnCount = resultSet.getMetaData().getColumnCount();
        Object[] rowData;
        
        if (columnCount < 2) {
            throw new SQLException("resultSetToHashMap: At least two columns needed for conversion.");
        }
        
        //Loop through the ResultSet data rows and adding to the HashMap the first column as the key,
        //and the second column as the value.
        while(resultSet.next()) {
            rowData = getRowData(resultSet, columnCount);
            map.put(rowData[0], rowData[1]);
        }
        
        
    }
    
    
    
    
    /**
     * The following method converts an ArrayList, containing string arrays of ResultSet row data,
     * into a two dimensional string array. This array must be first initialized before it
     * can be passed as an argument to the ArrayList toArray() method.
     *
     * The first dimension represents the number of rows, including the column names, and
     * can be obtained from the ArrayList size() method which returns the number of elements
     * in the ArrayList.
     *
     * The second dimension represents the number of columns. This value is obtained from the
     * length of the string array in the first element of the ArrayList since:
     *    1. The length of all the string arrays in the ArrayList must be the same.
     *    2. There must also be at least one row in the ArrayList representing the column names.
     */
    public String[][] arrayListToString(ArrayList arrayList) {
        //Monitor mon=start("arrayListToString");
        
        String[][] resultStrings = new String[arrayList.size()][((String[])arrayList.get(0)).length];
        arrayList.toArray(resultStrings);
        
        //mon.stop();
        return resultStrings;
    }
    
    /**
     * The following method simply takes the ResultSet and converts it to a two dimensional
     * array of strings containing the column names and data using calls to
     * the resultSetToArrayList and arrayListToString methods.
     */
    public String[][] resultSetToString(ResultSet resultSet) throws SQLException {
        Monitor mon=start("resultSetToString[][]");
        String[][] arr=null;
        try {
            ArrayList arrayList = new ArrayList();
            resultSetToArrayList(arrayList, resultSet);
            arr=arrayListToString(arrayList);
        }
        finally {
            mon.stop();
            return arr;
        }
    }
    
    /**
     * The following method simply takes the ResultSet and converts it to a two dimensional
     * array of strings containing the column names and data using calls to
     * the resultSetToArrayList and arrayListToString methods.
     */
    public String[][] fullResultSetToString(ResultSet resultSet) throws SQLException {
        ArrayList arrayList = new ArrayList();
        fullResultSetToArrayList(arrayList, resultSet);
        return arrayListToString(arrayList);
    }
    
    
    /**
     * The following method returns an array of strings containing the column names for
     * a given ResultSetMetaData object.
     */
    public String[] getColumnNames(ResultSetMetaData resultSetMetaData) throws SQLException {
        int columnCount = resultSetMetaData.getColumnCount();
        String columnNames[] = new String[columnCount];
        
        for(int colIndex=1; colIndex<=columnCount; colIndex++){
            columnNames[colIndex-1] = resultSetMetaData.getColumnName(colIndex);
        }
        
        return columnNames;
    }
    
    
    
    /**
     * The following method returns an array of strings containing the column names for
     * a given ResultSetMetaData object.  NEW CODE
     */
    public String[] getColumnNames(ResultSetMetaData resultSetMetaData, int[] returnCols) throws SQLException {
        int columnCount = returnCols.length; //resultSetMetaData.getColumnCount();
        String columnNames[] = new String[returnCols.length];
        
        for(int colIndex=0; colIndex<columnCount; colIndex++){
            columnNames[colIndex] = resultSetMetaData.getColumnName(returnCols[colIndex]);
        }
        
        return columnNames;
    }
    
    
    /**
     * The following method returns an array of objects containing the data values for
     * each column for the current row in a given ResultSet. The columnCount parameter
     * saves the method having to address the ResultSetMetaData to obtain the number of
     * columns.
     */
    private Object[] getRowData(ResultSet resultSet, int columnCount) throws SQLException {
        Object rowData[] = new String[columnCount];
        
        for(int colIndex=1; colIndex<=columnCount; colIndex++) {
            rowData[colIndex-1] = resultSet.getObject(colIndex);
        }
        return rowData;
    }
    
    
/*
public TabularResults.ResultSet toTabularResultSet(ResultSet resultSet) throws SQLException
{
    Monitor mon=start("toTabularResultSet()");
    TabularResults.ResultSet trs=null;
    try {
        trs = com.sybase.CORBA.jdbc11.IDL.getResultSet(resultSet);
    }
    finally
    {
        mon.stop();
        return trs;
    }
}
 
 
public ResultSet toResultSet(TabularResults.ResultSet tabularResultSet) throws SQLException
{
    Monitor mon=start("toResultSet()");
    ResultSet rs=null;
    try {
        rs=com.sybase.CORBA.jdbc11.SQL.getResultSet(tabularResultSet);
    } finally
    {
        mon.stop();
        return rs;
    }
 
}
 */
}

