package com.jamonapi.utils;

import java.util.*;

/** 
 * A NodeTree works with Compositenodes and LeafNodes to create heirarchical object trees.  An example is a file system where directories
 * can be viewed LeafNodes and directories can be viewed as CompoisteNodes.  TreeNodes can be viewed as the top level of the tree structures.
 *
 * For JAMon monitors are grouped in heirarchical with the CompositeNodes being MonitorComposite.  MonitorComposites can contain other 
 * MonitorComposites or LeafNodes which are the actual Monitors.   Monitors are identified by monitor lables which are strings seperated 
 * by a common delimiter for example "MyApp.jsp.homePage".  The first two levels of the heirarchy are CompositeNodes/MonitorComposites 
 * (MyApp and jsp), and the entry after the last delimiter is always the LeafNode (in this case a Monitor).
 * 
 * Note that NodeTrees/CompositeNodes/LeafNodes can represent ANY hierarchical/tree structure.  JAMon uses this generic pattern to
 * model Monitors.
 *
 * NodeTree's use the first non-alphanumeric to represent the token delimiter.  For example any combination of 
 * the following are considered token delimiters. (i.e. the following are all valid)
 *   pages.homepage (token delimeter is .)
 *   pages\homepage (token delimeter is \)
 *   pages\homepage.button (token delimeter is \ not .)
 *
 * note - Exception to the rule that token delimeters are the first non-alphanumeric characters are - and _. 
 * They are considered part of the variable name.  A token delimeter can be any of the following:  !@#$%^&*()+=[]\\{}|;':\",./<>?`~
 *
 **/



public class NodeTree extends java.lang.Object {

    private CompositeNode compositeNodeFactory;
    
    public NodeTree(CompositeNode compositeNodeFactory)     {
        this.compositeNodeFactory=compositeNodeFactory;
    }

    /** Return the highest level CompositeNode in the NodeTree.  i.e. the entry point to all other CompositeNodes and LeafNodes. 
     *  To use the file system analogy then the root node would be the root directory in Unix or c:\ in Windows.
     **/
    public CompositeNode getRootNode()    {
        return compositeNodeFactory.getRootNode();
    }
    
    /** Return the NodeTree's LeafNode object that is identified by the locator and type.  Also add it to the 
     * NodeTree.  
     *
     * This method uses the StringTokenizer to parse the locator String.  StringTokenizer is notoriously slow
     * and this code could be replaced in the future.   
     *
     * Sample Call:
     *  LeafNode leafNode=nodeTree.getLeafNode("MyApp.jsp.homePage", "type");
     *
     * The variable "type" is can be any string that is useful to the developer when creating a LeafNode.
     *
     * Note:
     *  getLeafNode(...) recursively calls getCompositeNode(...) for each NodeTree in the label.  In the above example 
     *  getCompositeNode(...) would be called once for "MyApp" and once for "jsp".  
     *
     **/
    public LeafNode getLeafNode(String locator, String type)    {
        String nodeName;
        CompositeNode previousNode;
        StringTokenizer st=getStringTokenizer(locator);
        int lastToken=st.countTokens()-1;
        CompositeNode currentNode=getRootNode();
        
        
        // all nodes but the last one should be a composite
        for (int i=0; i<lastToken; i++) {
            previousNode = currentNode;
            nodeName     = st.nextToken();
            
            currentNode = previousNode.getCompositeNode(nodeName);// creates if it doesn't exist
            previousNode.addCompositeNode(nodeName, currentNode);
        }
        
        // add the leaf node.
        nodeName = st.nextToken();
        
        LeafNode leafNode = currentNode.getLeafNode(nodeName, type);
        
        currentNode.addLeafNode(nodeName, leafNode );
        return leafNode;
        
    }
    

   /** Return the NodeTree's CompositeNode object that is identified by the locator.  Also add it to the 
     * NodeTree.  
     *
     * This method uses the StringTokenizer to parse the locator String.  StringTokenizer is notoriously slow
     * and this code could be replaced in the future.   
     *
     * Sample Call:
     *  CompositeNode compositeNode=nodeTree.getCompositeNode("MyApp.jsp");
     *
     **/

    public CompositeNode getCompositeNode(String locator)    {
        StringTokenizer st=getStringTokenizer(locator);;
        CompositeNode currentNode=getRootNode();
        CompositeNode previousNode;
        
        while (st.hasMoreTokens()) {
            previousNode=currentNode;
            String nodeName=st.nextToken();
            
            currentNode=previousNode.getCompositeNode(nodeName);
            previousNode.addCompositeNode(nodeName, currentNode);
        }
        
        return currentNode;
        
    }
    
    
    private final boolean NODE_EXISTS=true;
    private final boolean NODE_NO_EXISTS=false;
    
    
    /** Determine if the CompositeNode represented by the locator string exists.
     * Sample Call:  
     *   if (nodeTree.compositeNodeExists("MyApp.jsp"))
     *      ...
     **/
    
    public boolean compositeNodeExists(String locator)    {
        
        StringTokenizer st=getStringTokenizer(locator);;
        CompositeNode currentNode=getRootNode();
        
        while (st.hasMoreTokens()) {
            String nodeName=st.nextToken();
            
            if (currentNode.compositeNodeExists(nodeName))
                currentNode = currentNode.getCompositeNode(nodeName);
            else
                return NODE_NO_EXISTS;
            
        }
        
        return NODE_EXISTS;
        
    }
    
    
    /** Determine if the leaf represented by the locator string exists.
     * Sample Call:  
     *   if (nodeTree.leafNodeExists("MyApp.jsp.homePage"))
     *      ...
     **/
    public boolean leafNodeExists(String locator)    {
        String nodeName;
        StringTokenizer st=getStringTokenizer(locator);;
        int lastToken=st.countTokens()-1;
        CompositeNode currentNode=getRootNode();
        
        // all nodes but the last one should be a composite
        for (int i=0; i<lastToken; i++) {
            nodeName=st.nextToken();
            if (currentNode.compositeNodeExists(nodeName))
                currentNode=currentNode.getCompositeNode(nodeName);
            else
                return NODE_NO_EXISTS;
        }
        
        // check that the last node is a leaf.
        nodeName=st.nextToken();
        
        return currentNode.leafNodeExists(nodeName);
        
    }

    /** Determine if the Node represented by the locator string exists.  The node can be either a CompositeNode or LeafNode.
     * Sample Call:  
     *   if (nodeTree.nodeExists("MyApp.jsp.homePage"))
     *      ...
     **/
    
    public boolean nodeExists(String locator)    {
        return (compositeNodeExists(locator) || leafNodeExists(locator));
    }
    
    
    private StringTokenizer getStringTokenizer(String locator)    {
        return new StringTokenizer(locator, getDelimiter(locator));
    }
    
    // return the first non alpha numeric character and use it as a node delimiter
    // i.e. if pages.home is passed then "." is returned or if pages/home is passed then "/" is returned
    // if no delimiter is passed i.e. "pages" then an empty string will be used as the delimiter.
    private String getDelimiter(String locator)    {
        for (int i=0; i<locator.length(); i++) {
            if (isTokenDelimiter(locator.charAt(i)))
                return String.valueOf(locator.charAt(i));
        }
        
        return "";
    }
    
    
    
    private  boolean isTokenDelimiter(char ch)    {
        //the following are valid delimiters
        // !@#$%^&*()+=[]\\{}|;':\",./<>?`~
        return !(Character.isLetterOrDigit(ch) || Character.isWhitespace(ch) );
    }
    
    
    
    // add your data members here
}

