package com.jamonapi;

import com.jamonapi.utils.*;
import java.util.*;

/** This is the Monitor class that starts the decorator chain of Monitors.  It tracks the time for this monitor and passes the time
 *  down the chain to the other Monitors, so they can track the other statistics.
 **/
public class TimingMonitor extends BaseMonitor implements LeafNode {
    private long startTime = 0;
    AccumulateMonitorInterface childMonitor;
    boolean isRunningFlag = true;
    
    long accrued;
    boolean hasChildren;  // used to determine how to execute toString()

        
    TimingMonitor()     {
        this(NullAccumulateMonitor.createInstance());
        hasChildren=false;
    }
    
    
    TimingMonitor(AccumulateMonitorInterface childMonitor)     {
        this.childMonitor=childMonitor;
        hasChildren=true;
    }
    
    /** Returns the time that the Monitor has been running **/
    public long getAccrued()     {
        return accrued + timeElapsedSinceLastStart();
    }
    
    private long timeElapsedSinceLastStart()     {
        if (isRunning()) {
            return System.currentTimeMillis() - startTime;
        }
        else {
            return 0;
        }
    }
    
    /** Resets the accrued time and restarts the Monitor **/
    public void reset()    {
        accrued=0;
        resetStartTime();
    }
    
    /** increase the time by the specified ammount of milliseconds.  Typically this will not be called. **/
    public void increase(long increaseValue)      {
        if (isRunning()) {
            childMonitor.increase(increaseValue);
            accrued += increaseValue;
        }
    }
    
    // This class doesn't need to be thread safe as whenever start() is called a new TimingMonitor is returned
    
    /** Start timing for the Monitor **/
    public Monitor start()     {
        TimingMonitor mon=new TimingMonitor(childMonitor);
        
        mon.resetStartTime();
        mon.hasChildren=hasChildren;
        mon.childMonitor.start();
         
        return mon;
    }
    
    
    public void getData(ArrayList rowData)     {
        childMonitor.getData(rowData);
    }
    
    public void getHeader(ArrayList header)     {
        childMonitor.getHeader(header);
    }
    

    /** Stop the Monitor and keep track of how long it was running. Call the next Monitor in the Monitor decorator chain **/
    public Monitor stop()     {
        increase(timeElapsedSinceLastStart());
        
        if (isRunning())
            childMonitor.stop();
        
        setRunning(false);
        
        return this;
        
    }
    
    private boolean primary=false;
    
    public boolean isPrimary() {
        return primary;
    }
    
    public void setPrimary(boolean primary) {
        this.primary=primary;
        childMonitor.setPrimary(primary);
    }
    
    
    private boolean isRunning()     {
        return isRunningFlag;
    }
    
    private void resetStartTime()     {
        startTime = System.currentTimeMillis();
    }
    
    public String toString()     {
        String message = getAccruedString() + " ms.";
        
        if (hasChildren)
            message+=" ("+childMonitor.toString()+")";
        
        return message;
    }
    
    
    private void setRunning(boolean flag)     {
        isRunningFlag=flag;
    }
    
    /** Method with the classes test code **/
    public static void main(String[] args) throws Exception     {
        Monitor m = new TimingMonitor();
        m=m.start();
        Thread.sleep(1000);
        m.stop();
        System.out.println("should be about 1000="+m.getAccrued());
        System.out.println("should be same as above "+m.getAccrued());
        Thread.sleep(1000);
        System.out.println("should be same as above "+m.getAccrued());
        m.reset();
        Thread.sleep(10);
        System.out.println("should be 0="+m.getAccrued());
        m=m.start();
        Thread.sleep(100);
        System.out.println("should be about 100="+m.getAccrued());
        
        m=m.start();
        Thread.sleep(500);
        m.stop();
        m=m.start();
        Thread.sleep(250);
        System.out.println("should be about 250="+m.getAccrued());
        
        m=m.start();
        Thread.sleep(500);
        //m.pause();
        //System.out.println("should be about 500="+m.getAccrued()); // pause doesn't reset the accumulator
        //m=m.start();
        Thread.sleep(250);
        System.out.println("should be about 750="+m.getAccrued());
        System.out.println("toString should be about 750="+m.stop());
        Thread.sleep(250);
        
        System.out.println("toString should be about 750="+m.stop());
        
        for (int i=1; i<=5; i++) {
            m=m.start();
            Thread.sleep(i*10);
            m.stop();
        }
        
        System.out.println("\ntesting start() and stop() in loop.  should be about 50="+m);
        
        m.reset();
        System.out.println("\nreset to 0's: "+m);
        
        
    }
}

