package com.jamonapi;


import java.util.*;

/** Monitor that keeps track of various timing statistics such as max, min, average, hits, total, and standard deviation.**/

public class TimeStatsMonitor extends AccumulateMonitor {
    private long min=Integer.MAX_VALUE;
    private long max=Long.MIN_VALUE;
    private int  hits;
    private long total;
    private long sumOfSquares;
    
    
    /*
    The std deviation caclulation that does not require keeping all data points.  xi below stands for the ith
    data point.
     
    required info to calculate:
     
        n - count of all x's
        sum (xi^2) - must add the newest xi data point squared to this totoal each time - sumOfSquares
        sum(xi) - must add the newest xi data point to this total each time - total
     
        numerator=sum(xi^2)    -    (sum(xi)^2)/n;
        std_dev=square_root(numerator/(n-1));
     */
    
    
    public TimeStatsMonitor()     {
        super();
    }
    
    public TimeStatsMonitor(AccumulateMonitorInterface childMonitor)     {
        super(childMonitor);
    }
    

    /** Called when increase() is called by the stop() method.  This is the method that keeps track of the various statistics being tracked
     **/
    synchronized protected void increaseThis(long value)     {
        // calculate min
        if (value < min)
            min = value;
        
        // calculate max
        if (value > max)
            max = value;
        
        // calculate hits i.e. n
        hits++;
        
        // calculate total i.e. sumofX's
        total+=value;
        
        sumOfSquares+=value*value;
        
    }
    
    /** Reset all variables for this object.  The effect of this is to reset this objects variables to the state they were in when the object
     *  was first created.
     **/
    synchronized protected void resetThis()     {
        min=Long.MAX_VALUE;
        max=Long.MIN_VALUE;
        total=sumOfSquares=hits=0;
    }
    
    
    /** Make a String representation of this object.   Information displayed in the string include hits, average, total, min and max **/
    synchronized protected String toStringThis()     {
        return  getDisplayString(HITS, convertToString(hits), NONE) +
        getDisplayString(AVG, convertToString(avg()), MILLISECONDS) +
        getDisplayString(TOTAL, convertToString(total), MILLISECONDS) +
        //getDisplayString(STANDARD_DEVIATION, convertToString(stdDev()), MILLISECONDS) +
        getDisplayString(MIN, convertToString(min), MILLISECONDS) +
        getDisplayString(MAX, convertToString(max), MILLISECONDS);
    }
    
    
    private long avg()     {
        if (hits==0)
            return 0;
        else
            return total/hits;
    }
    
    private long stdDev()    {
        long stdDeviation=0;
        if (hits!=0)          {
            long sumOfX=total;
            int n=hits;
            int nMinus1= (n<=1) ? 1 : n-1;  // avoid 0 divides;
            
            long numerator = sumOfSquares-((sumOfX*sumOfX)/n);
            stdDeviation=(long) java.lang.Math.sqrt(numerator/nMinus1);
        }
        
        return stdDeviation;
    }
    
    
    
    synchronized protected void getDataThis(ArrayList rowData)    {
        rowData.add(convertToString(hits));
        rowData.add(convertToString(avg()));
        rowData.add(convertToString(total));
        rowData.add(convertToString(stdDev()));
        rowData.add(convertToString(min));
        rowData.add(convertToString(max));
        
    }
    
    
    protected void getHeaderThis(ArrayList header)    {
        header.add(HITS+" "+NONE);
        header.add(AVG+" "+MILLISECONDS);
        header.add(TOTAL+" "+MILLISECONDS);
        header.add(STANDARD_DEVIATION+" "+MILLISECONDS);
        header.add(MIN+" "+MILLISECONDS);
        header.add(MAX+" "+MILLISECONDS);
    }
    
    
    /** Returned the total accrued time for this monitor **/
    synchronized public long getAccrued()    {
        return total;
    }
    
    
    
    /** The main method contains this classes test code **/
    public static void main(String[] args) throws Exception    {
        TimeStatsMonitor mon=new TimeStatsMonitor();
        
        System.out.println("should not have 0 divide error="+mon);
        for (int i=1; i<5; i++)
            mon.increase(i);
        
        mon.increase(100);
        
        System.out.println("toString()="+mon);
        
    }
}

