package com.jamonapi;

import java.util.*;

/** TimeStatsDistMonitor keeps track of statistics for JAMon time ranges.  See www.jamonapi.com for more info.
 * One powerful feature of this class is that it correlates performance to the number of active monitors with this label,
 * the number of primary active monitors and the number of globally active monitors.
 **/

public class TimeStatsDistMonitor extends AccumulateMonitor {
    static private Counter allActive=new Counter();  // tracks the number of ALL monitors that are active
    static private Counter primaryActive=new Counter(); // tracks the number of primary monitors that are active
    
    private int active=0;
    
    
    public TimeStatsDistMonitor()     {
        super();
        populateDist();
    }
    
    public TimeStatsDistMonitor(AccumulateMonitorInterface childMonitor)     {
        super(childMonitor);
        populateDist();
    }
    
    
    /** increase(...) is called by TimingMonitor right before stop is called.  The distribution statistics are tracked in this method **/
    synchronized protected void increaseThis(long value)     {
        distribution(value);
    }
    
    
    /** Increment counters for the number of active monitors of this instance, globally active monitors and primary active monitors.  These
     *  numbers can be associated to performance to give a feel for scalability and other performance relationships in the application.
     *  see www.jamonapi.com for more info on the active concept and JAMon time ranges
     **/
    synchronized protected void startThis()     {
        // Counts how many total monitors are currently in progress as a rough check of how busy the system is
        // these need to be synchronized because they can be called from other objects start methods so they wouldn't
        // by syncrhonized by the startThis() method
        allActive.increment();
        if (isPrimary())
            primaryActive.increment();
        
        active++;
        
    }
    
    /** Decrement counters for the number of active monitors of this instance, globally active monitors and primary active monitors.  These
     *  numbers can be associated to performance to give a feel for scalability and other performance relationships in the application.
     *  see www.jamonapi.com for more info on the active concept and JAMon time ranges
     **/
    synchronized protected void stopThis()     {
        if (active>0)
            active--;
        
        if (isPrimary())  // see should i use ints or longs.  what about checking for>0 before decrement??????
            primaryActive.decrement();
        
        allActive.decrement();
    }
    
    
    // Index constants for array referencing
    private static final int i10=0;
    private static final int i20=1;
    private static final int i40=2;
    private static final int i80=3;
    private static final int i160=4;
    private static final int i320=5;
    private static final int i640=6;
    private static final int i1280=7;
    private static final int i2560=8;
    private static final int i5120=9;
    private static final int i10240=10;
    private static final int i20480=11;
    private static final int iLAST_GROUP=12;
    private static final int DIST_SIZE=13;
    
    private FrequencyDist[] dist;
    
    private void populateDist()     {
        dist=new FrequencyDist[DIST_SIZE];
        
        for (int i=0; i<DIST_SIZE; i++)
            dist[i]=new FrequencyDist();
    }
    
    // Note because distribution is always accessed via a synchronized method it doesn't have to be synchronized.
    private void distribution(long value)     {
        // calculates hits, avg, and global active for the distribution to which the passed arg 'value' belongs
        if (value>=0)
            getDist(value).increment(value);
        else
            throw new RuntimeException("Value passed to TimeStatsMonitor.distribution(value) must be positive. It was="+value);
        
    }
    
    
    // This method is passed the number of milliseconds representing this monitors accrued time.  This method makes sure
    // the statistics we are tracking are put in the proper time range
    private FrequencyDist getDist(long value) {
        if (value<=10)
            return dist[i10];
        else if (value<=20)
            return dist[i20];
        else if (value<=40)
            return dist[i40];
        else if (value<=80)
            return dist[i80];
        else if (value<=160)
            return dist[i160];
        else if (value<=320)
            return dist[i320];
        else if (value<=640)
            return dist[i640];
        else if (value<=1280)
            return dist[i1280];
        else if (value<=2560)
            return dist[i2560];
        else if (value<=5120)
            return dist[i5120];
        else if (value<=10240)
            return dist[i10240];
        else if (value<=20480)
            return dist[i20480];
        else
            return dist[iLAST_GROUP];
        
    }
    
    synchronized protected void resetThis()     {
        populateDist();
    }
    
    
    synchronized protected String toStringThis()     {
        return  "";
    }
    
    
    synchronized protected void getDataThis(ArrayList rowData)    {
        rowData.add(isPrimary() ? "Yes" : "&nbsp");
        
        for (int i=0; i<DIST_SIZE; i++)
            rowData.add(dist[i].toString());
        
    }
    
    protected void getHeaderThis(ArrayList header)    {
        final String SUFFIX="ms.";// Hits/Avg&nbsp(Avg Active/Primary Active/Global Active)";
        
        header.add("Primary");
        header.add("0-10"+SUFFIX);
        header.add("11-20"+SUFFIX);
        header.add("21-40"+SUFFIX);
        header.add("41-80"+SUFFIX);
        header.add("81-160"+SUFFIX);
        header.add("161-320"+SUFFIX);
        header.add("321-640"+SUFFIX);
        header.add("641-1280"+SUFFIX);
        header.add("1281-2560"+SUFFIX);
        header.add("2561-5120"+SUFFIX);
        header.add("5121-10240"+SUFFIX);
        header.add("10241-20480"+SUFFIX);
        header.add(">20480"+SUFFIX);
        
    }
    
    
    
    // Note becauset FrequencyDis is always accessed via a synchronized method it doesn't have to be synchronized.
    // This class tracks statistics for each JAMon time range.  Statistics include hits, total execution time, and the
    // activity statistics.
    final class FrequencyDist     {
        private int   hits;
        private long  totalTime; // used to calculate avg time for this distribution
        private long  allActiveTotal;  // used to calculate the average active total monitors for this distribution
        private long  primaryActiveTotal;
        private int   activeTotal;
        
        private void increment(long value)         {
            hits++;
            totalTime+=value;
            
            allActiveTotal+=allActive.count;  // total of all monitors active
            primaryActiveTotal+=primaryActive.count;  // total of all monitors marked primary as active
            activeTotal+=active;  // total of this monitor active
        }
        
        private float avgTotalActive()         {
            // avg active of all monitors active when this method was called
            if (hits==0)
                return 0f;
            else
                return (float) allActiveTotal/hits;
        }
        
        private float avgActive()         {
            // avg active of this monitor when this method was called
            if (hits==0)
                return 0f;
            else
                return (float) activeTotal/hits;
        }
        
        
        private float avgPrimaryActive()         {
            // avg active of all monitors marked as primary when this method was called
            if (hits==0)
                return 0f;
            else
                return (float) primaryActiveTotal/hits;
        }
        
        private long avgTime()         {
            if (hits==0)
                return 0;
            else
                return totalTime/hits;
        }
        
        public String toString()        {
            final String SPACE="&nbsp";
            if (hits==0)
                return SPACE;
            else
                return  convertToString(hits)+"/"+
                convertToString(avgTime())+SPACE+"("+
                convertToString(avgActive())+"/"+
                convertToString(avgPrimaryActive())+"/"+
                convertToString(avgTotalActive())+")";
        }
    }
    
    
    /** Test method for this class **/
    public static void main(String[] args) throws Exception    {
        TimeStatsDistMonitor mon=new TimeStatsDistMonitor();
        
        System.out.println("There should be 1 entry for each distribution range");
        System.out.println("All entries should have 5/low value of range/3");
        
        mon.start();
        mon.start();
        mon.start();
        
        for (int i=1; i<=5; i++) {
            mon.increase(0);
            mon.increase(20);
            mon.increase(40);
            mon.increase(80);
            mon.increase(160);
            mon.increase(320);
            mon.increase(640);
            mon.increase(1280);
            mon.increase(2560);
            mon.increase(5120);
            mon.increase(10240);
            mon.increase(20480);
            mon.increase(50000);
        }
        
        
        mon.stop();
        mon.stop();
        mon.stop();
        
        ArrayList rowData=new ArrayList();
        mon.getData(rowData);
        
        for(int i=0; i<rowData.size(); i++)
            System.out.println("distribution index "+i+"="+rowData.get(i));
        
        
    }
}

