package com.jamonapi;

import java.util.*;
import com.jamonapi.utils.*;

/**
 * MonitorFactory - Class that controls monitors including creating, starting, enabling, disabling and resetting them.  
 * This class uses the Gang of 4's creational patterns.  Monitors returned by MonitorFactory are thread safe.
 *
 * @author Steve Souza
 * @version 1.01
 */
public class MonitorFactory extends java.lang.Object {
    private static MonitorFactoryEnabled  enabledFactory=new MonitorFactoryEnabled();
    private static MonitorFactoryDisabled disabledFactory=new MonitorFactoryDisabled();
    
    private static boolean debugEnabled=true;
    private static boolean enabled=true;
    private static int debugPriorityLevel=0;
    
    

    /** Returns the composite monitor specified in the argument.  Composite Monitor's are groups of other monitors
     * (including other composites). <br>
     *
     * <p><code>Sample Call: String html=MonitorFactory.getComposite("pages").getReport(); </code></p>
     *
     * @param locator A string that locates the Composite Monitor.
     * @return MonitorComposite
     */    
    public static MonitorComposite getComposite(String locator)   {
        return getFactory().getComposite(locator);
    }
    
    /** Returns the factory for creating debug monitors.  The debug factory can be disabled independently from the
     * regular factory.  Debug monitors are no different than monitors returned by the regular monitor factory.
     * However the debug factory can be used to monitor items in a test environment and disable them in production.<br>
     *
     * <p><code>Sample Call: MonitorFactory.getDebugFactory().start();</code></p>
     *
     * @return MonitorFactoryInterface
     */        
    public static MonitorFactoryInterface getDebugFactory()     {
        return getFactory(isDebugEnabled());
    }
    
   /**
    * Returns the factory for creating debug monitors.  The debug factory can be disabled independently from the 
    * regular factory.  Debug monitors are no different than monitors returned by the regular monitor factory.  
    * However the debug factory can be used to monitor items in a test environment and disable them in production.<br>
    * This method takes a priority level threshold as an argument.  If the passed priority level is <b>greater than or equal</b> to the
    * MonitorFactory debug priority level then a Monitor is returned or else a null Monitor is returned (no statistics will be kept).<br>
    * 
    * <p><code>Sample Call: MonitorFactory.getDebugFactory(20).start();<br>  
    * if MonitorFactory debug priority level is 10 then a non null Monitor will be returned</code></p>
    *
    * @return  MonitorFactoryInterface 
    */        
    public static MonitorFactoryInterface getDebugFactory(int _debugPriorityLevel)     {
        return getFactory(isDebugEnabled(_debugPriorityLevel));
    }
    
    /**
     * Returns all gathered statistics as an HTML table as a String.  This can be displayed in a JSP or Servlet.<br>
     *
     * <p><code>Sample Call:  String html=MonitorFactory.getReport();</code></p>
     *
     **/
    public static String getReport() throws Exception     {
        return getRootMonitor().getReport();
    }
    
    /**
     * Returns gathered statistics underneath lower in the heirarchy than the locator string.  
     * The returned String is as an HTML table as a String.  This can be displayed in a JSP or Servlet.<br>
     *
     * <p><code>Sample Call:  String html=MonitorFactory.getReport("MyApplication.DataAccess");<br>
     * This would return statistics for MyApplication.DataAccess.open(), MyApplication.DataAccess.close(),...</code></p>
     *
     **/
    public static String getReport(String locator) throws Exception     {
        return getComposite(locator).getReport();
    }
    
    
    /** Returns the topmost Composite Monitor **/
    public static MonitorComposite getRootMonitor()     {
        return getFactory().getRootMonitor();
    }


    /**
     * Wipes out all statistics that have been gathered. Subsequent calls to the start(...) methods will continue to gather statistics.<br>
     *
     * <p><code>Sample Call:  MonitorFactory.reset();</code></p>
     **/
    public static void reset()     {
        enabledFactory=new MonitorFactoryEnabled();
    }
    
    /**
     * Enable or disable the debug factory.  The debug factory can be enabled/disabled at runtime.  Calling this method with a false
     * also disables calls to MonitorFactory.getDebugFactory(int debugPriorityLevel)<br>
     *
     * <p><code>Sample Call:<br>
     *  MonitorFactory.setDebugEnabled(false);<br>
     *  MonitorFactory.getDebugFactory().start(); // no stats are gathered.<br>
     *  MonitorFactory.getDebugFactory(20).start(); // no stats are gathered.<br></code></p>
     */
    public static void setDebugEnabled(boolean _debugEnabled)     {
        debugEnabled=_debugEnabled;
    }
    
    /**
     * Enable or disable the priority driven debug factory.  The debug factory can be enabled/disabled at runtime.  
     * Calling this method with a false has no effect on MonitorFactory.getDebugFactory()<br>
     *
     * <p><code>Sample Call:<br>
     *  MonitorFactory.setDebugEnabled(false);<br>
     *  MonitorFactory.getDebugFactory(20).start(); // no stats are gathered.<br></code></p>
     */
    public static void setDebugPriorityLevel(int _debugPriorityLevel)     {
        debugPriorityLevel=_debugPriorityLevel;
    }

    /**
     * Enable or disable the factory.  The factory can be enabled/disabled at runtime.  Calling this method with a false
     * also disables calls to both MonitorFactory.getDebugFactory(), and MonitorFactory.getDebugFactory(int debugPriorityLevel)<br>
     *
     * <p><code>Sample Call:<br>
     *  MonitorFactory.setEnabled(false);<br>
     *  MonitorFactory.start(); // no stats are gathered.<br>
     *  MonitorFactory.start("MyApp.DataAccess.open()"); // no stats are gathered.<br>
     *  MonitorFactory.getDebugFactory().start(); // no stats are gathered.<br>
     *  MonitorFactory.getDebugFactory(20).start(); // no stats are gathered.<br></code></p>
     */
    public static void setEnabled(boolean _enabled)     {
        enabled=_enabled;
    }
    
    /** Call this method if you don't want to use the default name or location for JAMonAdmin.jsp that is returned in the JAMon report.
     **/
    public static void setJAMonAdminPage(String JAMonAdminPage) {
        MonitorConverter.setJAMonAdminPage(JAMonAdminPage);
    }
    
    /** Return a Monitor and begin gathering timing statistics for it.<br>
     *
     * <p><code>Sample Call:<br>
     *  Monitor mon=MonitorFactory.start("MyApp.DataAccess.open()");<br>
     *  ...code being monitored...<br>
     *  System.out.println(mon.stop());<br></code></p>
     **/
    public static Monitor start(String locator)   {
        return getFactory().start(locator);
    }
    
    /** Return a Monitor and begin gathering timing statistics for it.  Statistics for this start() method will not be returned by 
     * MonitorFactory.getReport()<br>
     *
     * <p><code>Sample Call:<br>
     *  Monitor mon=MonitorFactory.start();<br>
     *  ...code being monitored...<br>
     *  System.out.println(mon.stop());<br></code></p>
     **/
    public static Monitor start()     {
        return getFactory().start();
    }
    
    /** Return a Monitor and begin gathering timing statistics for it.  See the online JAMon users manual for an explanation of this 
     *  method.<br>
     *
     * <p><code>Sample Call:<br>
     *  Monitor mon=MonitorFactory.startPrimary("MyApp.jsp.HomePage");<br>
     *  ...code being monitored...<br>
     *  System.out.println(mon.stop());<br></code></p>
     **/
    public static Monitor startPrimary(String locator)   {
        return getFactory().startPrimary(locator);
    }
    
    
    
    
    //******* Protected methods
    protected static MonitorFactoryInterface getFactory()     {
        return getFactory(isEnabled());
    }
    
    protected static MonitorFactoryInterface getFactory(boolean _enabled)     {
        if (_enabled)
            return enabledFactory;
        else
            return disabledFactory;
    }
    
   
    // Note for debug to be considered enabled the factory must also be enabled.
    protected static boolean isDebugEnabled()     {
        return debugEnabled && isEnabled();
    }
    
    protected static boolean isDebugEnabled(int _debugPriorityLevel)     {
        // for a given debug level to be enabled all of the following must be true
        // - the factory must be enabled.
        // - debugging must be enabled
        // - the priority level requested must be withing the acceptable range
        return  _debugPriorityLevel>=debugPriorityLevel && isDebugEnabled();
        
    }
    
    protected static boolean isEnabled()     {
        return enabled;
    }


    protected static void setNodeTree(NodeTree monitorTree)   {
        enabledFactory.setNodeTree(monitorTree);
    }
    
     
   
    
    /*
    Inner classes used for enabled and disabled factories.   They both inherit from MonitorFactoryBase.
    The classes are inner because they are an implementation detail for this class.
    */
    static abstract class MonitorFactoryBase implements MonitorFactoryInterface   {
        protected MonitorComposite rootMonitor;  // Parent to all other MonitorComposites
        
        public Monitor start(String locator)    {
            return createInstance(locator, MonitorLeafFactory.DEFAULT).start();
        }
        
        public Monitor startPrimary(String locator)    {
            return createInstance(locator, MonitorLeafFactory.PRIMARY).start();
        }
        
        abstract protected Monitor createInstance(String locator, String type);
           
        
    }
    
    /* Factory used when a factory is enabled */
    static class MonitorFactoryEnabled extends MonitorFactoryBase  {
        
        // monitorTree is used to create the summary monitors the first time they are executed.  After the monitors creation
        // existingLeafNodes is used instead.
        private NodeTree monitorTree;
        //??? may need to synchronize iteration seperately.   
        private Map existingLeafNodes=Collections.synchronizedMap(AppMap.createInstance());  
        private TimingMonitor simpleMonitor=new TimingMonitor();  // used for monitors that don't save summary statistics
        
        protected MonitorFactoryEnabled()    {
            rootMonitor = new MonitorComposite();
            monitorTree = new NodeTree(rootMonitor);
        }
        
        public void setNodeTree(NodeTree monitorTree)    {
            this.monitorTree=monitorTree;
        }
                
        private boolean monitorExists(String locator)    {
            return existingLeafNodes.containsKey(locator);
        }
        
        private Monitor getMonitor(String locator)    {
            // a new TimingMonitor is given to the requestor, however the same childMonitors are used.
            // This allows for the accumulation of statistics.
            Monitor mon = (TimingMonitor) existingLeafNodes.get(locator);
            return mon;
            
        }
        
        private Monitor createMonitor(String locator, String type)    {
            Monitor mon;
            
            synchronized(this)          {
                // gets the Monitor from node tree map if it exists or creates the Monitor if it doesn't exist
                mon = (TimingMonitor) monitorTree.getLeafNode(locator, type);
                existingLeafNodes.put(locator, mon);
            }
            
            return mon;
        }
        
        
        
        // If the monitor already exists get it from the existingLeafNodes, else create it via the slower NodeTree
        protected Monitor createInstance(String locator, String type)     {
            if (monitorExists(locator))
                return getMonitor(locator);
            else
                return createMonitor(locator, type);
        }
        
        public Monitor start()     {
            return simpleMonitor.start();
        }
        
        
        public MonitorComposite getComposite(String locator)    {
            // it is a programming error to get a nonexisting composite.  composites are created when
            // when of the creation methods are called (start(...) and createInstance())
            if (!monitorTree.compositeNodeExists(locator))
                throw new RuntimeException("The requested MonitorComposite does not exist: "+locator);
            
            return (MonitorComposite) monitorTree.getCompositeNode(locator);
        }
        
        
        public MonitorComposite getRootMonitor()    {
            return rootMonitor;
        }
        
    }
    
    // When a MonitorFactory is disabled it simply returns a NullMonitor (see Martin Fowler's excellent refactoring book)
    // A null object is an object that provides the required interface, but no implementation.  So when Monitor's are disabled
    // the code still executes however it completes very quickly because the code doesn't do anything.
    static class MonitorFactoryDisabled extends MonitorFactoryBase {
        
        private Monitor nullMonitor = new NullMonitor();
        
        protected Monitor createInstance(String locatory, String type) {
            return nullMonitor;
        }
        
        public Monitor start()     {
            return nullMonitor;
        }
        
        public MonitorComposite getComposite(String locator)    {
            return new MonitorComposite();
        }
        
        public MonitorComposite getRootMonitor()    {
            return new MonitorComposite();
        }
                
    }
    
        
    /** Test code for the MonitorFactory class.
     **/
    public static void main(String[] args) throws Exception {
        
        
        Monitor testMon=null;
        
        for (int i=0; i<50; i++) {
            testMon=MonitorFactory.start("pages/homepage");
            Thread.sleep(10);
            testMon.stop();
            Thread.sleep(10);
            
        }
        
        System.out.println("\nComposite test1 should be 10 ms., total 500 ms., 50 hits: "+testMon.stop());
        
        for (int i=0; i<50; i++) {
            testMon=MonitorFactory.start("pages.homepage");
            Thread.sleep(10);
            testMon.stop();
            Thread.sleep(10);
        }
        
        System.out.println("\nComposite test2 (Should be double the previous numbers): "+testMon.stop());
        System.out.println("\ngetting root monitor: "+getRootMonitor());
        
    }
}

