package com.jamonapi;

import java.util.*;
import com.jamonapi.utils.*;

/** MonitorConverter is a utility class used by MonitorComposite to convert Monitor data to various formats.  **/


public class MonitorConverter extends java.lang.Object implements MonitorReportInterface {
    private ResultSetUtils utils=ResultSetUtils.createInstance();  // Generic conversion class.
    private MonitorComposite rootComposite;
    private static String JAMON_ADMIN_PAGE_DEFAULT="JAMonAdmin.jsp";
    private String JAMonAdminPage=JAMON_ADMIN_PAGE_DEFAULT;
    private static String[][] EMPTY_ARRAY={{""}};
    
    private ArrayList rowsList = new ArrayList();
    
    // used for initial sizing of the ArrayList that holds the data returned by MonitorComposite
    static private int TYPICAL_NUM_CHILDREN=30;
    
    /** Constructor that takes the MonitorComposite object that the Converter methods will be called against. **/
    protected MonitorConverter(MonitorComposite rootComposite)   {
        this.rootComposite=rootComposite;
    }
    
    /** Constructor that takes the MonitorComposite object that the Converter methods will be called against. **/
    protected MonitorConverter(MonitorComposite rootComposite, String JAMonAdminPage)   {
        this.rootComposite=rootComposite;
        this.JAMonAdminPage=JAMonAdminPage;
    }
    
    /** Return an html table in String format that is sorted by the passed column in ascending or descending order **/
    public String getReport(int sortCol, String sortOrder) throws Exception {
        return new MonitorReport(sortCol, sortOrder).getReport();
    }
    
    
    /** Return an html table in String format in the default sort order **/
    public String getReport() throws Exception   {
        return getReport(1, "asc");
    }
    
    
    
    /***** start inner class MonitorReport - class that displays the monitor html report *****/
    private class MonitorReport {
        
        private final int sortCol;
        private final String sortOrder;
        private final StringBuffer table=new StringBuffer(20000);
        
        public MonitorReport(int sortCol, String sortOrder) {
            this.sortCol=sortCol;
            this.sortOrder=sortOrder;
        }
        
        public String getReport() throws Exception   {
            addReportHeader();
            addReportBody();
            
            return table.toString();
        }
        
        
        private void addReportHeader() {
            table.append("<!-- Begin Report Sect. -->\n");
            table.append("<table border='0' cellpadding='0' cellspacing='0'>\n");
            table.append("<tr>\n");
            table.append("<td>\n");
            table.append("<table class='layoutmain' border='1' cellpadding='2' cellspacing='0' rules='all'>\n");
            table.append("<tr class='headtextr' valign='top'>\n");
            
            
            String[] header=MonitorComposite.getHeader();
            int rows=header.length;
            String currentHeaderCell=null;
            String lastHeaderCell=null;
            
            for (int i=0; i<rows; i++) {
                int sortIndex=i+1;
                if (sortIndex==sortCol) {  // if this header entry is the one that has been sorted on
                    String newSortOrder = ("asc".equalsIgnoreCase(sortOrder)) ? "desc" : "asc";
                    currentHeaderCell = "<th><a href='"+JAMonAdminPage+"?sortCol="+sortIndex+"&sortOrder="+newSortOrder+"'>"+header[i]+"</a><br><img src='"+sortOrder+".gif'></th>\n";
                }
                else {
                    currentHeaderCell = "<th><a href='"+JAMonAdminPage+"?sortCol="+sortIndex+"&sortOrder=desc'>"+header[i]+"</a><br></th>\n";
                }
                
                if (i==0) // repeat the first header at the end so the report is easier to read.
                    lastHeaderCell=currentHeaderCell;
                
                
                table.append(currentHeaderCell);
            }
            
            table.append(lastHeaderCell+"</tr>\n");
            table.append("<tr class='headtextr'>\n");
            table.append("<th colspan='13'>&nbsp;</th>\n");
            table.append("<th colspan='13'>Hits/Avg ms. &nbsp &nbsp &nbsp &nbsp &nbsp (Avg Active/Primary Active/Global Active)</th>\n");
            table.append("<th>&nbsp;</th>\n</tr>\n");
            
        }
        
        
        private void addReportBody() {
            Object[][] data=getData("", sortCol-1, sortOrder);
            
            int rows=data.length;
            int cols=data[0].length;
            
            for (int i=0; i<rows; i++)   {
                // note row i=0 is row 1
                String oddOrEven = (i%2 == 0) ? "odd" : "even";
                String monitorLabel = data[i][0].toString();
                
                table.append("<tr class='"+oddOrEven+"' onMouseOver='rollOnRow(this, \"Statistics for "+monitorLabel+"\")' onMouseOut='rollOffRow(this)'>\n");
                String labelEntry="<th class='headtextc' align='left' nowrap>"+monitorLabel+"</th>\n";
                table.append(labelEntry);
                
                for (int j=1; j<cols; j++) {
                    table.append("<td nowrap>"+data[i][j]+"</td>\n");
                }
                
                table.append(labelEntry);  // repeat the label column to make the report more readable
                table.append("</tr>\n");
            }
            
            table.append("</table>\n<!-- End Report Sect. -->\n");
            
        }
    }
    
    /***** end inner class MonitorReport *****/
    
    
    public String[][] getData()    {
        return getData("");
    }
    
    
    public String[][] getData(String label)   {
        getCompositeData(label, rootComposite);
        
        int rows=rowsList.size();
        
        if (rows==0) {
            return EMPTY_ARRAY;
        }
        else
            return utils.arrayListToString(rowsList);
        
    }
    
    public Object[][] getData(String label, int sortCol, String sortOrder) {
        Object[][] data=getData(label);
        
        if (data==EMPTY_ARRAY) // Avoid index out of range exception when monitor has no data
            return data;
        else
            return Misc.sort(data, sortCol, sortOrder);
    }
    
    
    
    
    protected void getData(String label, MinimalMonitor monitor)  {
        if (monitor instanceof MonitorComposite)
            getCompositeData(label, (MonitorComposite) monitor);  // recursive
        else
            getLeafData(label, monitor);
    }
    
    
    protected void getCompositeData(final String localLabel, final MonitorComposite composite)   {
        // inner classes in a method can only access local variables that are defined as final
        class GetMonitorData implements Command {
            Map.Entry mapEntry;
            MinimalMonitor monitor;
            
            public void execute(Object value) throws Exception {
                String label=localLabel;
                mapEntry = (Map.Entry) value;
                
                monitor = (MinimalMonitor) mapEntry.getValue();
                String key=mapEntry.getKey().toString();  // something like pagesC or homepageL
                label += composite.getLabelFromKey(key);  // pages. or homepage
                
                getData(label, monitor);  // recursive
            }
        }
        
        composite.iterateMapEntries(new GetMonitorData());
    }
    
    
    
    
    protected void getLeafData(String label, MinimalMonitor monitor)   {
        ArrayList rowData=new ArrayList(TYPICAL_NUM_CHILDREN);// the arraylist will be created by the called routine.
        rowData.add(label);  // add display label such as "pages.homepage"
        monitor.getData(rowData);  // appends monitor data to the arraylist
        
        String[] rowDataArray=new String[0];
        
        rowsList.add(rowData.toArray(rowDataArray));
        
    }
    
    protected static void setJAMonAdminPage(String JAMonAdminPage) {
        MonitorConverter.JAMON_ADMIN_PAGE_DEFAULT=JAMonAdminPage;
    }

}

